#include "blob.h"
#include "mbed.h"

extern Serial pc;

// Take in a blob and determine whether it is inside the world.
// If the blob has escaped the world, put it back on the edge
// of the world and negate its velocity so that it bounces off
// the boundary. Use WORLD_WIDTH and WORLD_HEIGHT defined in "misc.h"
void BLOB_constrain2world(BLOB* b) {
    // ***
    // checking for x-based constraints
    if ((b->posx - b->rad) <= 0 || (b->posx + b->rad) >= WORLD_WIDTH) {
        b->vx = -1 * b->vx;
        if (b->posx < 0) {
            b->posx = b->rad;
        } else if (b->posx > WORLD_WIDTH) {
            b->posx = WORLD_WIDTH - b->rad;
        }
    }
    // checking for y-based constraints
    if ((b->posy - b->rad) <= 0 || (b->posy + b->rad) >= WORLD_HEIGHT)
        b->vy = -1 * b->vy;
        if (b->posy < 0) {
            b->posy = b->rad;
        } else if (b->posy > WORLD_HEIGHT) {
            b->posy = WORLD_HEIGHT - b->rad;
        }
}

// Randomly initialize a blob's position, velocity, color, radius, etc.
// Set the valid flag to true and the delete_now flag to false.
// delete_now is basically the derivative of valid. It goes true for one
// fram when the blob is deleted, and then it is reset to false in the next frame
// when that blob is deleted.
void BLOB_init(BLOB* b) {
    // ***
    // get a random radius
    int rad = (rand() % 5) + 1;
    BLOB_init(b, rad);
}

// Randomly initialize a blob. Then set the radius to the provided value.
void BLOB_init(BLOB* b, int rad) {
    // ***
    // using food color
    BLOB_init(b, rad, FOOD_COL);
}

// Randomly initialize a blob. Then set the radius and color to the
// provided values.
void BLOB_init(BLOB* b, int rad, int color) {
    // ***
    // get random (x,y,) for blob pos
    int genSpace_x = WORLD_WIDTH  - (3*rad);
    int genSpace_y = WORLD_HEIGHT - (3*rad);

    int rand_x = (rand() % genSpace_x) + rad;
    int rand_y = (rand() % genSpace_y) + rad;

    b->posx = rand_x;
    b->posy = rand_y;
    b->old_x = rand_x;
    b->old_y = rand_y;
    b->vx = ACC_THRESHOLD * 1000;      
    b->vy = ACC_THRESHOLD * 1000;      
    b->rad = rad;
    b->color = color;
    b->valid = 1;
    b->delete_now = 0;
}

// For debug purposes, you can use this to print a blob's properties to your computer's serial monitor.
void BLOB_print(BLOB b) {
    pc.printf("cur(%.3f, %.3f) old(%.3f, %.3f) vel<%.3f, %.3f> Col: 0x%x\n", b.posx, b.posy, b.old_x, b.old_y, b.vx, b.vy, b.color);
}

// Return the square of the distance from b1 to b2
float BLOB_dist2(BLOB b1, BLOB b2) {
    // ***
    float x1 = b1.posx;
    float y1 = b1.posy;

    float x2 = b2.posx;
    float y2 = b2.posy;

    return ((x2-x1) *(x2-x1)) + ((y2-y1) * (y2-y1));
}
