#include "mbed.h"
#include "Serial485.h"
#include "fsl_uart_hal.h"

const PinMap Serial485::PinMap_UART_RTS[] = {
    {PTE3 , UART_1, 3},
    {PTA3 , UART_0, 2},
    {PTA17, UART_0, 3},
    {PTB2 , UART_0, 3},
    {PTC1 , UART_1, 3},
    {PTC12, UART_4, 3},
    {PTC18, UART_3, 3},
    {PTD0 , UART_2, 3},
    {PTD4 , UART_0, 3},
    {NC  ,  NC    , 0}
};

Serial485::Serial485(PinName tx, PinName rx, PinName rts):
    RawSerial(tx, rx),
    rts(rts)
{
    if(rts != NC)
    {
        //find UART peripheral number
        uint32_t uart_rts = pinmap_peripheral(rts, PinMap_UART_RTS);
        if(uart_rts == (uint32_t)NC)
            error("pinmap not found for peripheral");
        else
        {
            uart_instance = uart_rts;
            enableRS485();
        }
    }
}

Serial485::~Serial485()
{
    if(rts != NC)
        disableRS485();
}

void Serial485::enableRS485()
{
    uart_hal_configure_transmitter_rts_polarity(uart_instance, 1);
    uart_hal_enable_transmitter_rts(uart_instance);
    pinmap_pinout(rts, PinMap_UART_RTS);
    pin_mode(rts, PullDown);
}

void Serial485::disableRS485()
{
    uart_hal_disable_transmitter_rts(uart_instance);
    pin_function(rts, 0);
    pin_mode(rts, PullUp);
}

void Serial485::set9thBit(uint8_t value)
{
    BW_UART_C3_T8(uart_instance, value);
}

void Serial485::puts9(const uint16_t *data, uint16_t count)
{
    while(count-- > 0)
    {
        while(!writeable());
        putc9(*data++);
    }
}

void Serial485::putc9(uint16_t data)
{
    uart_hal_putchar9(uart_instance, data);
}

uint16_t Serial485::getc9()
{
    uint16_t data;
    while(!readable());
    uart_hal_getchar9(uart_instance, &data);
    return data;
}

uint32_t Serial485::getBaudRate()
{
    return _baud;
}
