/**
 *  @file   FEP.h
 *  @brief  FEPライブラリ
 *  @author 安澤瑠
 *  @date   21/10/15
 */
#ifndef FEP_H
#define FEP_H

#define TIMEOUT_COUNT 10

#include "mbed.h"

/**
 *  @class myFEP Class for communicating using FEP-01, FEP-02
 *  @brief Class for communicating using FEP-01, FEP-02
 *  @note  mbed-os 6 では使えません。  Not compatible with mbed-os 6
 */
class myFEP : public RawSerial {
public :
    /** constructor
     *  @param tx   FEPと接続するSerialTX pin
     *  @param rx   FEPと接続するSerialRX pin
     *  @param addr 通信相手のアドレス
     *  @param baud 通信速度(デフォルト115200)
     */
    myFEP(PinName tx, PinName rx, uint8_t addr_, int baud=115200);

    /** Start receiving
     */
    void StartReceive();
    
    /** Check timeout
     * @brief 0.1秒毎のループで受信のタイムアウトをチェック
     */
    void TimeoutLoop();

    /** Interrupt input
     */
    void ReceiveBytes();

    /** extract the message
     */
    void CheckData();

    /** Write the received message
     *  @param data 受信メッセージを格納する配列
     */
    uint8_t GetData(uint8_t *data);

    /** send message
     *  @brief data配列のデータをlength分送信する
     *  @param data   送るデータ配列のアドレス
     *  @param length 送るデータのバイト数
     *
     *  @return 0  通信成功
     *  @return 1  データ量過多
     *  @return 2  コマンドエラー
     *  @return 3  通信相手からのレスポンス無し
     *  @return 4  通信相手が受け取りに失敗
     *  @return -1 可否不能
     */
    int8_t SendData(uint8_t *data, uint8_t length);

    /** Response acquisition function
     *  @return 0  通信成功
     *  @return 2  コマンドエラー
     *  @return 3  通信相手からのレスポンス無し
     *  @return 4  通信相手が受け取りに失敗
     *  @return -1 可否不能
     */
    int8_t GetResponse();
    
    bool status; //! (TIMEOUT_COUNT * 0.1)秒間通信が来ていないと0

private :
    
    Ticker timeoutTimer;

    uint8_t addr;         //! 通信相手のアドレス
    uint8_t buffer[256];  //! 全受信データを格納するリングバッファ配列
    uint8_t retdata[128]; //! メッセージのみを格納する配列
    uint8_t bufindex;     //! buffer の添え字変数
    uint8_t retindex;     //! retdata の添え字変数
    uint8_t length;       //! メッセージバイト数
    uint8_t timeout;      //! 0.1秒間通信が成功していないと1増える
};

#endif