/** Interface to control a stepper motor.
 * Specially design for Arduino motor shield, which has a L298 H-brigde.
 *
 * First version is written by Peter Ilsoe.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef ArduinoMotorShield_H
#define ArduinoMotorShield_H

#include "mbed.h"

class ArduinoMotorShield
{

public:

    /** Create a DC/Stepper motor control interface. This class is able to control either DC motor or a Stepper motor.
     * This is specially designed for the Arduino motor shield board
     *
     * @param numberOfSteps The number of steps of the stepper motor used.
     * @param brakeA A DigitalOut, set high when the motor should brake.
     * @param brakeB A DigitalOut, set high when the motor should brake.
     * @param dirA A DigitalOut, set high when the DC motor A should go forward, set low when going backwards. For the stepper motor the logic will control this.
     * @param dirB A DigitalOut, set high when the DC motor B should go forward, set low when going backwards. For the stepper motor the logic will control this.
     * @param pvmA A DigitalOut/PVM, Depending on what mode (DC motor or stepper motor) this pin will become PVM(DC motor) or digital output(stepper).
     * @param pvmB A DigitalOut/PVM, Depending on what mode (DC motor or stepper motor) this pin will become PVM(DC motor) or digital output(stepper).
     */
    ArduinoMotorShield(int numberOfSteps, PinName brakeA, PinName brakeB, PinName dirA, PinName dirB, PinName pvmA, PinName pvmB);

    /** Set the speed of a Stepper motor
     *
     * @param whatSpeed The speed of the Stepper motor as a RPM
     */
    void setStepperSpeed(long whatSpeed);

    /** Move the stepper a number of steps
     *
     * @param numberOfSteps The number of steps. If the number is negative,
     * the motor moves in the reverse direction.
     */
    void step(int numberOfSteps);

    /** Run at a specific speed until a startStop becomes false.
     *
     * @param startStop If true the stepper motor will run at the specified speed. If false stepper motor will stop
     */
    void runStepperAtSpeed(bool startStop, int whatSpeed, int direction);
    
    /** Call back member for timer tick
     *
     */
    void runMotor(void);

private:
    void stepMotor(int this_step);

    int direction; // Direction of rotation
    int speed; // Speed in RPMs
    unsigned long step_delay; // delay between steps, in ms, based on speed
    int number_of_steps; // total number of steps this motor can take
    int step_number; // which step the motor is on
    PinName digiPvm; // store the common pin for PVM when DC motor and digital pin when stepper motor

    DigitalOut _BRAKEA;
    DigitalOut _BRAKEB;
    DigitalOut _DIRA;
    DigitalOut _DIRB;
    DigitalOut _ENA_A;
    DigitalOut _ENA_B;
    long last_step_time; // time stamp in ms of when the last step was taken

};


#endif
