/* USBHostPhs.h */
/* Modified by 2015 phsfan
 *  for ABIT SMA-01
 */
/* mbed USBHost Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* WANDongle.h, WANDongleSerialPort.h */
/* Copyright (c) 2010-2012 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef _USBHostPhs_H_
#define _USBHostPhs_H_

#include "USBHost.h"
#include "IUSBHostSerial.h"

#include "rtos.h"

#define PHS_VID 0x075E
#define PHS_PID 0x0028

#define WANDONGLE_MAX_OUTEP_SIZE 64
#define WANDONGLE_MAX_INEP_SIZE 64

/**
 * A class to communicate a USB virtual serial port
 */
class USBHostPhsPort : public IUSBHostSerial {
public:
    /**
    * Constructor
    */
    USBHostPhsPort();

    enum IrqType {
        RxIrq,
        TxIrq
    };

    enum Parity {
        None = 0,
        Odd,
        Even,
        Mark,
        Space
    };

    void connect(USBHost* _host, USBDeviceConnected * _dev,
        uint8_t _serial_intf, USBEndpoint* _bulk_in, USBEndpoint* _bulk_out);

    /** Set the baud rate of the serial port
     *
     *  @param baudrate The baudrate of the serial port (default = 9600).
     */
    void baud(int baudrate = 921600);

    /** Set the transmission format used by the Serial port
     *
     *  @param bits The number of bits in a word (default = 8)
     *  @param parity The parity used (USBHostSerialPort::None, USBHostSerialPort::Odd, USBHostSerialPort::Even, USBHostSerialPort::Mark, USBHostSerialPort::Space; default = USBHostSerialPort::None)
     *  @param stop The number of stop bits (1 or 2; default = 1)
     */
    void format(int bits = 8, Parity parity = USBHostPhsPort::None, int stop_bits = 1);

    virtual int getc();
    virtual int putc(int c);
    virtual int readPacket();
    virtual int writePacket();
    virtual int readable();
    virtual int writeable();
    virtual void attach(IUSBHostSerialListener* pListener);
    virtual void setupIrq(bool en, IUSBHostSerial::IrqType irq = IUSBHostSerial::RxIrq);

protected:
    void reset();

private:
    USBHost * host;
    USBDeviceConnected * dev;

    USBEndpoint * bulk_in;
    USBEndpoint * bulk_out;
    uint32_t size_bulk_in;
    uint32_t size_bulk_out;

    void init();

    uint8_t buf[64];

    typedef struct {
        uint32_t baudrate;
        uint8_t stop_bits;
        uint8_t parity;
        uint8_t data_bits;
    } PACKED LINE_CODING;

    LINE_CODING line_coding;

    void rxHandler();
    void txHandler();

    uint8_t serial_intf;


    uint8_t buf_out[WANDONGLE_MAX_OUTEP_SIZE];
    volatile uint32_t buf_out_len;
    uint32_t max_out_size;
    volatile bool lock_tx;
    volatile bool cb_tx_en;
    volatile bool cb_tx_pending;
    Mutex tx_mtx;

    uint8_t buf_in[WANDONGLE_MAX_INEP_SIZE];
    volatile uint32_t buf_in_len;
    volatile uint32_t buf_in_read_pos;
    volatile bool lock_rx;
    volatile bool cb_rx_en;
    volatile bool cb_rx_pending;
    Mutex rx_mtx;

    IUSBHostSerialListener* listener;

};


class USBHostPhs : public IUSBEnumerator, public USBHostPhsPort
{
public:
    USBHostPhs();

    /**
     * Try to connect a serial device
     *
     * @return true if connection was successful
     */
    bool connect();

    void disconnect();

    /**
    * Check if a any serial port is connected
    *
    * @returns true if a serial device is connected
    */
    bool connected();

protected:
    USBHost* host;
    USBDeviceConnected* dev;
    uint8_t port_intf;
    int ports_found;

    //From IUSBEnumerator
    virtual void setVidPid(uint16_t vid, uint16_t pid);
    virtual bool parseInterface(uint8_t intf_nb, uint8_t intf_class, uint8_t intf_subclass, uint8_t intf_protocol); //Must return true if the interface should be parsed
    virtual bool useEndpoint(uint8_t intf_nb, ENDPOINT_TYPE type, ENDPOINT_DIRECTION dir); //Must return true if the endpoint will be used

private:
    bool dev_connected;
};

#endif
