/* AbitUSBModem.cpp */
/* Modified by 2015 phsfan
 *  for ABIT SMA-01
 */
/* VodafoneUSBModem.cpp */
/* Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */


#define __DEBUG__ 0

#ifndef __MODULE__
#define __MODULE__ "AbitUSBModem.cpp"
#endif

#include "core/fwk.h"

#include "AbitUSBModem.h"
#include "Socket.h"

AbitUSBModem::AbitUSBModem () :
   m_dongle(),                         // Construct AbitUSBModem: USB interface with two serial channels to the modem (USBSerialStream objects)
   m_pppStream((IUSBHostSerial&)m_dongle), // PPP connections are managed via another serial channel.
   m_at(&m_pppStream),                  // Construct ATCommandsInterface with the AT serial channel
   m_sms(&m_at),                       // Construct SMSInterface with the ATCommandsInterface
   m_ppp(&m_pppStream, &m_pppStream, &m_at, false), // Construct PPPIPInterface with the PPP serial channel
   m_dongleConnected(false),           // Dongle is initially not ready for anything
   m_ipInit(false),                    // PPIPInterface connection is initially down
   m_smsInit(false),                   // SMSInterface starts un-initialised
   m_atOpen(false)                    // ATCommandsInterface starts in a closed state
{
}

int AbitUSBModem::connect (const char* user, const char* password) {

  if( !m_ipInit )
  {
    m_ipInit = true;
    m_ppp.init();
  }
  m_ppp.setup(user, password);

  int ret = init();
  if(ret)
  {
    return ret;
  }

  m_at.close(); // Closing AT parser
  m_atOpen = false; //Will need to be reinitialized afterwards

  DBG("Connecting PPP");

  ret = m_ppp.connect();
  DBG("Result of connect: Err code=%d", ret);
  return ret;
}

int AbitUSBModem::disconnect () {
  DBG("Disconnecting from PPP");
  int ret = m_ppp.disconnect();
  if(ret)
  {
    ERR("Disconnect returned %d, still trying to disconnect", ret);
  }

  return OK;
}



int AbitUSBModem::init()
{
  //DBG("Entering init method for the VodafoneUSBModem");
  if( !m_dongleConnected )
  {
    DBG("Dongle is not connected");

    m_dongle.connect();

    m_dongleConnected = true;
    bool detectConnectedModem = false;          // local variable to use to create a while loop that we can break out of - this is used to detect if we can see a modem or not
    
    while(!detectConnectedModem)
    {
        for (int x=0; x<100;x++)
        {
            DBG("Trying to connect the dongle");
            m_dongle.connect();
            if (m_dongle.connected())
            {
                DBG("Great the dongle is connected - I've tried %d times to connect", x);
                detectConnectedModem = true;        // OK we can break out this while loop now - the dongle has been connected
                break;                              // Break out of the for loop once the dongle is connected - otherwise try for a while more
            }
            Thread::wait(1000);
        }
        if (!detectConnectedModem)
        {
            // OK we got this far - so give up trying and let someone know you can't see the modem
            m_dongleConnected = false;                  // set the member variable of this object to false - so if we get called again we know we have to try to detect again
            ERR("There is no dongle pluged into the board, or the module does not respond. Is the module/modem switched on?");
            Thread:wait(1000);
            //DBG("Last ditch attempt to re-initialise the USB Subsystem");
            //m_dongle.init();
            return HARDWARE_NO_RESPONSE;
        }
    }
  }

  if(m_atOpen)
  {
    return OK;
  }
  
  DBG("Starting AT thread if needed");
  int ret = m_at.open();
  if(ret)
  {
    return ret;
  }
  
  DBG("Sending initialisation commands");
  ret = m_at.init();
  if(ret)
  {
    return ret;
  }
/*
  ret = m_at.executeSimple("ATE0", NULL);
  DBG("Result of command: Err code=%d", ret);
  if(ret != OK)
  {
    return NET_PROTOCOL;
  } 
*/
  m_atOpen = true;

  return OK;
}


int AbitUSBModem::sendSM(const char* number, const char* message)
{
  int ret = init();
  if(ret)
  {
    return ret;
  }

  if(!m_smsInit)
  {
    ret = m_sms.init();
    if(ret)
    {
      return ret;
    }
    m_smsInit = true;
  }

  ret = m_sms.send(number, message);
  if(ret)
  {
    return ret;
  }

  return OK;
}

int AbitUSBModem::getSM(char* number, char* message, size_t maxLength)
{
  int ret = init();
  if(ret)
  {
    return ret;
  }

  if(!m_smsInit)
  {
    ret = m_sms.init();
    if(ret)
    {
      return ret;
    }
    m_smsInit = true;
  }

  ret = m_sms.get(number, message, maxLength);
  if(ret)
  {
    return ret;
  }

  return OK;
}

char* AbitUSBModem::getIPAddress()
{
    return m_ppp.getIPAddress();
}
