/* AbitUSBModem.cpp */
/* Modified by 2015 phsfan
 *  for ABIT SMA-01
 */
/* VodafoneUSBModem.cpp */
/* Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */


#define __DEBUG__ 0

#ifndef __MODULE__
#define __MODULE__ "AbitUSBModem.cpp"
#endif

#include "core/fwk.h"

#include "AbitModemInterface.h"
#include "Socket.h"

AbitModemInterface::AbitModemInterface (PinName tx, PinName rx, PinName cts, PinName rts, PinName reset, int baud) :
   m_module(tx, rx),                         // Construct AbitModemInterface: Serial interface
#if !DEVICE_SERIAL_FC
   m_cts(cts),
   m_rts(rts),
#endif
   m_reset(reset),
   m_pppStream(m_module), // PPP connections are managed via another serial channel.
   m_at(&m_pppStream),                  // Construct ATCommandsInterface with the AT serial channel
   m_sms(&m_at),                       // Construct SMSInterface with the ATCommandsInterface
   m_ppp(&m_pppStream, &m_pppStream, &m_at, false), // Construct PPPIPInterface with the PPP serial channel
   m_moduleConnected(false),           // Dongle is initially not ready for anything
   m_ipInit(false),                    // PPIPInterface connection is initially down
   m_smsInit(false),                   // SMSInterface starts un-initialised
   m_atOpen(false)                    // ATCommandsInterface starts in a closed state
{
    m_reset.output();
    m_reset = 0;
    m_module.baud(baud);
#if DEVICE_SERIAL_FC
    if (rts != NC && cts != NC) {
        m_module.set_flow_control(Serial::RTSCTS, rts, cts);
    } else
    if (rts != NC && cts == NC) {
        m_module.set_flow_control(Serial::RTS, rts);
    } else
    if (rts == NC && cts != NC) {
        m_module.set_flow_control(Serial::CTS, cts);
    }
#else
    m_rts = 0;
    m_cts.mode(PullUp);
#endif
    Thread::wait(100);
    m_reset.input();
    m_reset.mode(PullUp);
}

int AbitModemInterface::connect (const char* user, const char* password) {

  if( !m_ipInit )
  {
    m_ipInit = true;
    m_ppp.init();
  }
  m_ppp.setup(user, password);

  int ret = init();
  if(ret)
  {
    return ret;
  }

  m_at.close(); // Closing AT parser
  m_atOpen = false; //Will need to be reinitialized afterwards

  DBG("Connecting PPP");

  ret = m_ppp.connect();
  DBG("Result of connect: Err code=%d", ret);
  return ret;
}

int AbitModemInterface::disconnect () {
  DBG("Disconnecting from PPP");
  int ret = m_ppp.disconnect();
  if(ret)
  {
    ERR("Disconnect returned %d, still trying to disconnect", ret);
  }

  return OK;
}



int AbitModemInterface::init()
{
  //DBG("Entering init method for the VodafoneUSBModem");
  if( !m_moduleConnected )
  {
    DBG("Dongle is not connected");

    m_moduleConnected = true;
        Thread::wait(3000);

  }

  if(m_atOpen)
  {
    return OK;
  }
  
  DBG("Starting AT thread if needed");
  int ret = m_at.open();
  if(ret)
  {
    return ret;
  }
  
  DBG("Sending initialisation commands");
  ret = m_at.init();
  if(ret)
  {
    return ret;
  }

  ret = m_at.executeSimple("ATE1", NULL);
  DBG("Result of command: Err code=%d", ret);
  if(ret != OK)
  {
    return NET_PROTOCOL;
  } 

  m_at.executeSimple("AT#S1", NULL);
  m_at.executeSimple("AT#B1", NULL);
  Thread::wait(5000);

  m_atOpen = true;

  return OK;
}


int AbitModemInterface::sendSM(const char* number, const char* message)
{
  int ret = init();
  if(ret)
  {
    return ret;
  }

  if(!m_smsInit)
  {
    ret = m_sms.init();
    if(ret)
    {
      return ret;
    }
    m_smsInit = true;
  }

  ret = m_sms.send(number, message);
  if(ret)
  {
    return ret;
  }

  return OK;
}

int AbitModemInterface::getSM(char* number, char* message, size_t maxLength)
{
  int ret = init();
  if(ret)
  {
    return ret;
  }

  if(!m_smsInit)
  {
    ret = m_sms.init();
    if(ret)
    {
      return ret;
    }
    m_smsInit = true;
  }

  ret = m_sms.get(number, message, maxLength);
  if(ret)
  {
    return ret;
  }

  return OK;
}

char* AbitModemInterface::getIPAddress()
{
    return m_ppp.getIPAddress();
}
