/* IOSerialStream.cpp */
/* Modified by 2015 phsfan
 *  for ABIT SMA-01
 */
/* Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#define __DEBUG__ 0 //Maximum verbosity
#ifndef __MODULE__
#define __MODULE__ "IOSerialStream.cpp"
#endif

#include "core/fwk.h"

#include <cstring>

#include "IOSerialStream.h"

IOSerialStream::IOSerialStream(mbed::RawSerial& serial) : m_serial(serial), m_serialTxFifoEmpty(true),
m_availableSphre(1), m_spaceSphre(1), m_inBuf(), m_outBuf()
{
  m_availableSphre.wait();
  m_spaceSphre.wait();
  //Attach interrupts
  m_serial.attach(this, &IOSerialStream::readable, mbed::RawSerial::RxIrq);
  m_serial.attach(this, &IOSerialStream::writeable, mbed::RawSerial::TxIrq);

#if defined(TARGET_LPC176X) || defined(TARGET_LPC2368)
//  m_serial.set_flow_control(mbed::RawSerial::RTSCTS, P0_22, p12);
    _uart = LPC_UART1;
#elif defined(TARGET_LPC408X)
    _uart = (LPC_UART1_TypeDef*)LPC_UART2;
#elif defined(TARGET_LPC11UXX)
    _uart = LPC_USART;
#else
    _uart = NULL;
#endif
}

/*virtual*/ IOSerialStream::~IOSerialStream()
{
  m_serial.attach(NULL, mbed::RawSerial::RxIrq);
  m_serial.attach(NULL, mbed::RawSerial::TxIrq);
}

//0 for non-blocking (returns immediately), osWaitForever for infinite blocking
/*virtual*/ int IOSerialStream::read(uint8_t* buf, size_t* pLength, size_t maxLength, uint32_t timeout/*=osWaitForever*/)
{
  DBG("Trying to read at most %d chars", maxLength);
  int ret = waitAvailable(timeout);
  if(ret)
  {
    WARN("Error %d while waiting for incoming data", ret);
    return ret;
  }
  int readLen = MIN( available(), maxLength );
  *pLength = readLen;
  setupReadableISR(false);
  while(readLen--)
  {
    m_inBuf.dequeue(buf);
    buf++;
  }
  setupReadableISR(true);
#if __DEBUG__ >= 5
  buf -= *pLength;
  printf(" - Read:");
  for (int i = 0; i < *pLength; i ++) {
    if (buf[i] >= 0x20 && buf[i] < 0x7f) {
      printf(" %c", buf[i]);
    } else {
      printf(" %02x", buf[i]);
    }
  }
  printf("\r\n");
#endif
  DBG("Read %d chars successfully", *pLength);
  return OK;
}

/*virtual*/ size_t IOSerialStream::available()
{
  setupReadableISR(false); //m_inBuf.available() is not reentrant
  size_t len = m_inBuf.available();
  setupReadableISR(true);
  return len;
}

/*virtual*/ int IOSerialStream::waitAvailable(uint32_t timeout/*=osWaitForever*/) //Wait for data to be available
{
  int ret;
  if(available()) //Is data already available?
  {
    while( m_availableSphre.wait(0) > 0 ); //Clear the queue as data is available
    return OK;
  }

  DBG("Waiting for data availability %d ms (-1 is infinite)", timeout);
  ret = m_availableSphre.wait(timeout); //Wait for data to arrive or for abort
  if(ret <= 0)
  {
    DBG("Timeout");
    return NET_TIMEOUT;
  }
  if(!available()) //Even if abort has been called, return that data is available
  {
    DBG("Aborted");
    return NET_INTERRUPTED;
  }
  DBG("Finished waiting");
  while( m_availableSphre.wait(0) > 0 ); //Clear the queue as data is available
  return OK;
}

/*virtual*/ int IOSerialStream::abortRead() //Abort current reading (or waiting) operation
{
  if( !available() ) //If there is data pending, no need to abort
  {
    m_availableSphre.release(); //Force exiting the waiting state; kludge to pass a RC directly
  }
  else
  {
    DBG("Serial is readable"); ;
  }
  return OK;
}

void IOSerialStream::setupReadableISR(bool en)
{
#if defined(TARGET_LPC176X) || defined(TARGET_LPC408X) || defined(TARGET_LPC2368) || defined(TARGET_LPC11UXX)
  if(en)
  {
    _uart->IER |= 1 << 0;
  }
  else
  {
    _uart->IER &= ~(1 << 0);
  }
#endif
}

void IOSerialStream::readable() //Callback from m_serial when new data is available
{
  do
  {
//    m_inBuf.queue(_uart->RBR); //FIXME mbed libraries this is an awful kludge
    m_inBuf.queue(m_serial.getc()); //FIXME mbed libraries this is an awful kludge
  } while(m_serial.readable());
  m_availableSphre.release(); //Force exiting the waiting state
}

//0 for non-blocking (returns immediately), osWaitForever for infinite blocking
/*virtual*/ int IOSerialStream::write(uint8_t* buf, size_t length, uint32_t timeout/*=osWaitForever*/)
{
  DBG("Trying to write %d chars", length);
  int ret = waitSpace(timeout);
  if(ret)
  {
    WARN("Error %d while waiting for space", ret);
    return ret;
  }
  DBG("Writing %d chars", length);
  setupWriteableISR(false);
  while(length)
  {
    m_outBuf.queue(*buf);
    buf++;
    length--;
    if(length && !space())
    {
      DBG("Waiting to write remaining %d chars", length);
      setupWriteableISR(true);
      ret = waitSpace(timeout);
      if(ret)
      {
        WARN("Error %d while waiting for space", ret);
        return ret;
      }
      setupWriteableISR(false);
    }
  }
  //If m_serial tx fifo is empty we need to manually tx a byte in order to trigger the interrupt
  if( m_outBuf.available() && m_serialTxFifoEmpty )
  {
    m_serialTxFifoEmpty = false;
    uint8_t c;
    m_outBuf.dequeue(&c);
    m_serial.putc((char)c);
    //_uart->THR = c; //FIXME awful kludge
  }
  setupWriteableISR(true);
  DBG("Write successful");
  return OK;
}

/*virtual*/ size_t IOSerialStream::space()
{
  setupWriteableISR(false); //m_outBuf.available() is not reentrant
  size_t len = CIRCBUF_SIZE - m_outBuf.available();
  setupWriteableISR(true);
  return len;
}

/*virtual*/ int IOSerialStream::waitSpace(uint32_t timeout/*=osWaitForever*/) //Wait for space to be available
{
  int ret;
  if(space()) //Is still space already left?
  {
    while( m_spaceSphre.wait(0) > 0); //Clear the queue as space is available
    return OK;
  }

  DBG("Waiting for data space %d ms (-1 is infinite)", timeout);
  ret = m_spaceSphre.wait(timeout); //Wait for space to be made or for abort
  if(ret <= 0)
  {
    DBG("Timeout");
    return NET_TIMEOUT;
  }
  if(!space()) //Even if abort has been called, return that space is available
  {
    DBG("Aborted");
    return NET_INTERRUPTED;
  }
  while( m_spaceSphre.wait(0) > 0); //Clear the queue as space is available
  return OK;
}

/*virtual*/ int IOSerialStream::abortWrite() //Abort current writing (or waiting) operation
{
  if( !space() ) //If there is space left, no need to abort
  {
    m_spaceSphre.release(); //Force exiting the waiting state
  }
  return OK;
}

void IOSerialStream::setupWriteableISR(bool en)
{
#if defined(TARGET_LPC176X) || defined(TARGET_LPC408X) || defined(TARGET_LPC2368) || defined(TARGET_LPC11UXX)
  if(en)
  {
    _uart->IER |= 1 << 1;
  }
  else
  {
    _uart->IER &= ~(1 << 1);
  }
#endif
}

void IOSerialStream::writeable() //Callback from m_serial when new space is available
{
  if(m_outBuf.isEmpty())
  {
    m_serialTxFifoEmpty = true;
  }
  else
  {
    m_serialTxFifoEmpty = false;
    while(m_serial.writeable() && !m_outBuf.isEmpty())
    {
      uint8_t c;
      m_outBuf.dequeue(&c);
      m_serial.putc((char)c);
      //_uart->THR = c; //FIXME awful kludge
    }
  }
  if(!m_outBuf.isFull())
  {
    m_spaceSphre.release(); //Force exiting the waiting state
  }
}
