/**********************************************************************
* Copyright (C) 2018 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/


#include "mbed.h"
//#include "max32630fthr.h"
#include "max32630hsp.h"
#include "MAX8614X.h"
#include "USBSerial.h"

//#define DEBUG_ON 1

void executeSha256(MAX8614X &m, uint32_t *challenge, unsigned int challengeLen, bool romID, uint32_t *response);
bool isTheChipAuthenicated(uint32_t *expectedResponse, uint32_t *chipResponse);

#define RESPONSE_LEN32 8

MAX32630HSP icarus(MAX32630HSP::VIO_1V8);
//    MAX32630FTHR mbed_board(MAX32630FTHR::VIO_1V8);
SPI spi(P5_1, P5_2, P5_0); /* mosi, miso, sclk */
DigitalOut cs(P5_3);
PinName interrupt_pin = P3_2;
// Virtual serial port over USB
USBSerial microUSB; 

DigitalOut rLED(LED1);
DigitalOut gLED(LED2);
DigitalOut bLED(LED3);
// main() runs in its own thread in the OS
// (note the calls to Thread::wait below for delays)
/**
* @brief Sample main program for SHA256 Authenication using the MAX86140
* @version 1.0000.0
*
* @details Sample main program for MAX86140 authenication.
* The prints are sent to the terminal window (9600, 8n1).
* The program issues challenges to the MAX86140 SHA256 athenicator.
* The responses are compared to the expected responses.
* To run the program, drag and drop the .bin file into the 
* DAPLINK folder. After it finishes flashing, cycle the power or 
* reset the board.
*/

int main()
{
    #define CHALLENGE32_LEN 5
    uint32_t challenge_A[CHALLENGE32_LEN] = {  // 160 bit, no ROM
        0x00000001,
        0x00000000,
        0x00000000,
        0x00000000,
        0x00000000
    };
    uint32_t expectedResponse_A_romID[RESPONSE_LEN32] = {  // 160 bit, no ROM
        0xf987d79a,
        0xeb778ec7,
        0x33861bc8,
        0x745d4082,
        0x921e02fe,
        0x727d93c8,
        0x218e53ee,
        0x904e0c6b
    };
    uint32_t challenge_1[CHALLENGE32_LEN] = {  // 160 bit
        0x5e813524,
        0x5663d609,
        0x998d7b0d,
        0x52128465,
        0xcd0de301
    };
    uint32_t expectedResponse_1_romID[RESPONSE_LEN32] = {  // 160 bit
        0xfe3f805f,
        0x0af066bc,
        0xda0f55e2,
        0x0020ac16,
        0x13fa3406,
        0x59d0e5ae,
        0x79def309,
        0x685981a3
    };

    uint32_t challenge_2[CHALLENGE32_LEN] = {  // 160 bit, no ROM
        0xd2aa84c5,
        0x7277f7e5,
        0xdb8fd612,
        0x96ce69f2,
        0x4ec57ae8
    };

    uint32_t expectedResponse_2_noRomID[RESPONSE_LEN32] = {  // 160 bit, no ROM
        0x1a7135a2,
        0x51b99ca8,
        0xbdd9245e,
        0xb9c8e758,
        0x770ec1c4,
        0x474ffcc7,
        0x828afe4d,
        0x9cb3de6d
    };
    uint32_t challenge_3[CHALLENGE32_LEN] = {  // 160 bit
        0x21202280,
        0xcc9d45aa,
        0xb8133e96,
        0xd653380d,
        0x2ad5dd6b
    };
    uint32_t expectedResponse_3_romID[RESPONSE_LEN32] = {  // 160 bit
        0xe7750b8d,
        0xe29d9279,
        0x7cdc7053,
        0xa9f92519,
        0xa1e59d93,
        0x19cd930d,
        0xfb0fc974,
        0x2da0781e
    };
    uint32_t challenge_4[CHALLENGE32_LEN] = {  // 160 bit, no ROM
        0x4c3c0aca,
        0x618abdf2,
        0x34d8b341,
        0x1289f378,
        0x65b60deb
    };
    uint32_t expectedResponse_4_noRomID[RESPONSE_LEN32] = {  // 160 bit, no ROM
        0xae740f91,
        0x8d6f2d58,
        0x7486b0ba,
        0xe9a84580,
        0xfe9ce593,
        0x58d66c7d,
        0xc993d165,
        0xe5ae5983
    };

    uint32_t chip_response[RESPONSE_LEN32];
    bool valid = 0;

    rLED = LED_OFF;
    gLED = LED_ON;
    bLED = LED_OFF;

    printf("\r\n\rmax86140 authenication software\r\n");
    MAX8614X m(spi,cs,interrupt_pin);
    m.init();

    //● Compare MAC from MAX86140 wth Host's precalculated MAC.
    executeSha256(m, challenge_A, CHALLENGE32_LEN, 1, chip_response);
    //● Check PASS or FAIL.
    valid = isTheChipAuthenicated(chip_response, expectedResponse_A_romID);
    if (valid)
        printf("\r\n             Challenge A passed\r\n\r\n");
    else
        printf("\r\n             Challenge A failed\r\n\r\n");

    //● Compare MAC from MAX86140 wth Host's precalculated MAC.
    executeSha256(m, challenge_1, CHALLENGE32_LEN, 1, chip_response);
    //● Check PASS or FAIL.
    valid = isTheChipAuthenicated(chip_response, expectedResponse_1_romID);
    if (valid)
        printf("\r\n             Challenge 1 passed\r\n\r\n");
    else
        printf("\r\n             Challenge 1 failed\r\n\r\n");

    executeSha256(m, challenge_2, CHALLENGE32_LEN, 0, chip_response);
    valid = isTheChipAuthenicated(chip_response, expectedResponse_2_noRomID);
    if (valid)
        printf("\r\n             Challenge 2 passed\r\n\r\n");
    else
        printf("\r\n             Challenge 2 failed\r\n\r\n");

    executeSha256(m, challenge_3, CHALLENGE32_LEN, 1, chip_response);
    valid = isTheChipAuthenicated(chip_response, expectedResponse_3_romID);
    if (valid)
        printf("\r\n             Challenge 3 passed\r\n\r\n");
    else
        printf("\r\n             Challenge 3 failed\r\n\r\n");

    executeSha256(m, challenge_4, CHALLENGE32_LEN, 0, chip_response);
    valid = isTheChipAuthenicated(chip_response, expectedResponse_4_noRomID);
    if (valid)
        printf("\r\n             Challenge 4 passed\r\n\r\n");
    else
        printf("\r\n             Challenge 4 failed\r\n\r\n");

    //● Disable SHA_EN bit ( Write 0 to SHA_EN bit).     
    m.writeRegister(MAX8614X::MAX8614X_SHA_CFG_REG, 0);
    while(1) {
        gLED = !gLED;
        wait(1.0);
    }
}

bool isTheChipAuthenicated(uint32_t *expectedResponse, uint32_t * chipResponse)
{
    int i;
    for (i = 0; i < RESPONSE_LEN32; i++) {
        if (expectedResponse[i] != chipResponse[i])
            return 0;
    }
    return 1;
}

void transformData(uint8_t *inData, uint8_t *outData, unsigned int challengeLen)
{
    int i, j, k;
    k = 0;
    for (i = 0; i < (challengeLen/4); i++) {
        for (j = 3; j >= 0; j--) {
            outData[j+(i*4)] = inData[k];
            k++;
        }
    }
}
void executeSha256(MAX8614X &m, uint32_t *challenge, unsigned int challengeLen, bool romID, uint32_t *response)
{
    int i, j, k;
    uint8_t macData[256];
    uint8_t xData[256];
    uint32_t x32Data[64];
    uint32_t tmpData;
    const unsigned int responseLen32 = 8;
    uint8_t data[5];

    //● Enable SHA_DONE Interrupt
    m.writeRegister(MAX8614X::MAX8614X_INT_ENABLE2_REG, MAX8614X::MAX8614X_IE_SHA_DONE_EN);

    m.writeRegister(MAX8614X::MAX8614X_INT_ENABLE1_REG, 0);  // Disable all other interrupts

    //- Enable SHA_EN bit.
    m.writeRegister(MAX8614X::MAX8614X_SHA_CFG_REG,MAX8614X::MAX8614X_SHACFG_SHA_EN);

    //● Write 160-bit random challenge value to RAM using registers MEM_IDX and MEM_DATA.
    // Enable Memory Write, Select Bank 0, address 0x00 to 0xFF
    m.writeRegister(MAX8614X::MAX8614X_MEMORY_CONTROL_REG, MAX8614X::MAX8614X_MEMCNTRL_WR_EN_MASK | MAX8614X::MAX8614X_MEMCNTRL_BANK0_MASK);

    for (i = 0; i < 5; i++) {
        x32Data[i] = challenge[i];
        for (j = 0; j <= 3; j++) {
            xData[(i*4)+j] = x32Data[i] & 0xFF;
            x32Data[i] = x32Data[i] >> 8;
            k++;
        }
    }

#ifdef DEBUG_ON
    printf("\r\n Raw Input Data\r\n\r\n");
    for (i = 0; i < challengeLen; i++) {
        printf("%08x\r\n", challenge[i]);
    }
    printf("\r\n");
#endif

    //transformData(challenge, xData, challengeLen);

#ifdef DEBUG_ON
    printf("\r\n Transformed Input Data\r\n\r\n");
    for (i = 0; i < challengeLen*4; i++) {
        if (!(i % 4))
            printf("\r\n ");
        printf("%02x", xData[i]);
    }
        printf("\r\n ");
#endif
        
    for (i = 0; i < (challengeLen*4); i++) {
        m.writeRegister(MAX8614X::MAX8614X_MEMORY_INDEX_REG, i);
        m.writeRegister(MAX8614X::MAX8614X_MEMORY_DATA_REG, xData[i]);
    //}
  }

    // The message block consists of a 160-bit secret, a 160-bit challenge and 192 bits of constant data. Optionally, the 64-bit
    // ROM ID replaces 64 of the 192 bits of constant data used in the hash operation. 16 bits out of the 160-bit secret and 16
    // bits of ROM ID are programmable–8 bits each in metal and 8 bits each in OTP bits
    //● Write command, with ROM ID (0x35) or without ROM ID (0x36), to SHA_CMD register
    if (romID) {
        m.writeRegister(MAX8614X::MAX8614X_SHA_CMD_REG, MAX8614X::MAX8614X_SHACMD_MAC_ROM_ID);
    }
    else
    {
        m.writeRegister(MAX8614X::MAX8614X_SHA_CMD_REG,MAX8614X::MAX8614X_SHACMD_MAC_NO_ROM_ID);
    }

    m.readRegister(MAX8614X::MAX8614X_SHA_CMD_REG, data, 1);
    //● Write 1 to SHA_START and 1 to SHA_EN bit.
    m.writeRegister(MAX8614X::MAX8614X_SHA_CFG_REG,MAX8614X::MAX8614X_SHACFG_SHA_EN | MAX8614X::MAX8614X_SHACFG_SHA_START);
    
    //● Wait for SHA_DONE.
    data[0] = 0;
    k = 0;
    while(!data[0] && k < 100) {
        m.readRegister(MAX8614X::MAX8614X_INT_STATUS2_REG, data, 1);
        k++;
    }
    // ● Read 256 MAC value from RAM using registers MEM_IDX and MEM_DATA.
#ifdef DEBUG_ON
    printf("\r\n Raw Output\r\n\r\n");
#endif
    for (i = 64; i < 64+32; i++) {
#ifdef DEBUG_ON
        if (!(i % 4))
            printf("\r\n ");
#endif
        m.writeRegister(MAX8614X::MAX8614X_MEMORY_INDEX_REG, i);
        m.readRegister(MAX8614X::MAX8614X_MEMORY_DATA_REG, data, 1);
        xData[i-64] = data[0];
#ifdef DEBUG_ON
        printf("%02x", data[0]);
#endif
    } 
#ifdef DEBUG_ON
    printf("%\r\n");
#endif

    transformData(xData,macData, 32);
#ifdef DEBUG_ON
    printf("\r\n Formatted Response\r\n\r\n");
    for (i = 0; i < responseLen32*4; i++) {
        if (!(i % 4))
            printf("\r\n ");
        printf("%02x", macData[i]);
    }
    printf("\r\n\r\n");
    
#endif
     // transform data to Little-Endian
    k = 0;
    for (i = 0; i < (responseLen32); i++) {
        x32Data[i] = 0;
        for (j = 3; j >= 0; j--) {
            x32Data[i] = x32Data[i] << 8;
            x32Data[i] |= macData[k] & 0xFF;
            k++;
        }
    }

#ifdef DEBUG_ON
    printf("\r\n Formatted Response\r\n\r\n");
#endif
    for (i = 0; i < responseLen32; i++) {
#ifdef DEBUG_ON
        printf(" %08X \r\n",x32Data[i]);
#endif
        response[i] = x32Data[i];
    }
}
