/*************************************************************************
 * Dragonfly Example program of SMS on Freescale sensor board
 */

#include "mbed.h"
#include "mtsas.h"
#include "MbedJSONValue.h"
#include "HTTPJson.h"
#include <string>
#include <sstream>
#include "x_nucleo_iks01a1.h"

#define HTTP_RX_BUFFER_SIZE 1024
char http_rx_buf[HTTP_RX_BUFFER_SIZE];

// Debug serial port
static Serial debug(USBTX, USBRX);

// MTSSerialFlowControl - serial link between processor and radio
static MTSSerialFlowControl* io;

// Cellular - radio object for cellular operations (SMS, TCP, etc)
Cellular* radio;

// APN associated with SIM card
// this APN should work for the AT&T SIM that came with your Dragonfly
static const std::string apn = "broadband";

// Phone number to send SMS messages to
// just change the x digits - the 1 needs to stay!
static const std::string phone_number = " NEED A NUMBER with a 1";

// Keys needed by the m2x API
// see more details here: https://m2x.att.com/developer/documentation/v2/overview
// m2x device ID
static const std::string m2x_device_id = "";

// M2X primary API key
static const std::string m2x_api_key = "";

/* Instantiate the expansion board */
static X_NUCLEO_IKS01A1 *mems_expansion_board = X_NUCLEO_IKS01A1::Instance(I2C_SDA, I2C_SCL);

/* Retrieve the composing elements of the expansion board */
static GyroSensor *gyroscope = mems_expansion_board->GetGyroscope();
static MotionSensor *accelerometer = mems_expansion_board->GetAccelerometer();
static MagneticSensor *magnetometer = mems_expansion_board->magnetometer;
static HumiditySensor *humidity_sensor = mems_expansion_board->ht_sensor;
static PressureSensor *pressure_sensor = mems_expansion_board->pt_sensor;
static TempSensor *temp_sensor1 = mems_expansion_board->ht_sensor;
static TempSensor *temp_sensor2 = mems_expansion_board->pt_sensor;

// variables for sensor data
char streamAcc[] = "acc_rms";
char streamMag[] = "mag_rms";
char streamGyr[] = "gyr_rms";
char streamTemp[] = "temp_c";
char streamPres[] = "pressure";
char streamHum[] = "humidity";
char streamAccel[] = "accel";

// misc variables
static int sms_interval_ms = 60000;
static int read_interval_ms = 5000;
static int print_interval_ms = 1500;
int debug_baud = 115200;

bool radio_ok = false;

/****************************************************************************************************
// function prototypes
 ****************************************************************************************************/
bool init_mtsas();
char* httpResToStr(HTTPResult res);

/****************************************************************************************************
// main
 ****************************************************************************************************/
int main()
{
    mts::MTSLog::setLogLevel(mts::MTSLog::TRACE_LEVEL);
    debug.baud(debug_baud);
    logInfo("starting...");

    /****************************************************************************************************
          Initialize I2C Devices ************
     ****************************************************************************************************/

    int32_t accel_data[3];
    float accel_sensitivity;
    //float mag_data[3];
    //float gyro_data[3];
    
    float dataX = 0.0f;
    float dataY = 0.0f;
    float dataZ = 1.0f;
    float humidity = 0.0f;
    float temperature = 0.0f;
    float pressure = 0.0f;

// Initialization Radio Section **********************************************************

    radio_ok = init_mtsas();
    if (! radio_ok)
        logError("MTSAS init failed");
    else
        logInfo("MTSAS is ok");

//End Radio Initialization Section **********************************************************

    Timer sms_timer;
    sms_timer.start();
    Timer read_timer;
    read_timer.start();         // Timer data is set in the Variable seciton see misc variables    Timer motion_timer;
    Timer print_timer;
    print_timer.start();
    Timer motion_timer;
    motion_timer.start();
    
// http object initialization
    // HTTPClient object used for HTTP requests.
    HTTPClient http;
    
    // Enable strict certificate validation.
    // http.setPeerVerification(VERIFY_PEER);
    
    // Load certificates defined in ssl_certificates.h.
    // See comments in ssl_certificates.h for information on how to get and format root certificates.
//    if (http.addRootCACertificate(ssl_certificates) != HTTP_OK)
//        logError("loading SSL certificates failed");

    while (true) {
        if (read_timer.read_ms() > read_interval_ms) {

            accelerometer->get_x_axes(accel_data);
            accelerometer->get_x_sensitivity(&accel_sensitivity);
            dataX = accel_data[0] * accel_sensitivity;
            dataY = accel_data[1] * accel_sensitivity;
            dataZ = accel_data[2] * accel_sensitivity;
            humidity_sensor->get_humidity(&humidity);
            temp_sensor1->get_temperature(&temperature);
            pressure_sensor->get_pressure(&pressure);
            read_timer.reset();
        }
// send to m2x on same interval as printing
        if (print_timer.read_ms() > print_interval_ms) {

            logDebug("SENSOR DATA");
            logDebug("DataZ  %0.3f", dataZ);
            logDebug("DataY  %0.3f", dataY);
            logDebug("DataX  %0.3f", dataX);
            logDebug("Humidity: %0.2f", humidity);
            logDebug("Temperature: %0.1f", temperature);
            logDebug("Presure: %f", pressure);
            
            HTTPResult res;
            // IHTTPDataIn object - will contain data received from server.
            HTTPText http_rx(http_rx_buf, HTTP_RX_BUFFER_SIZE);
            
            std::ostringstream json_packet;
            json_packet << "{\"values\": {\"value\" : " \
                    << temperature << "}}";
            
            // IHTTPDataOut object - contains data to be posted to server.
            // HTTPJson automatically adds the JSON content-type header to the request.
            char *tx_string = new char[json_packet.str().length()];
            sprintf(tx_string, json_packet.str().c_str());
            HTTPJson http_tx(tx_string, json_packet.str().length());
            std::string m2x_header = "X-M2X-KEY: " + m2x_api_key + "\r\n";
            std::string url = "https://api-m2x.att.com/v2/devices/" + \
                                m2x_device_id + \
                                "/streams/temperature/values";
            http.setHeader(m2x_header.c_str());
            
            http.post(url.c_str(), http_tx, &http_rx);
            delete tx_string;
            if (res != HTTP_OK)
                logError("HTTPS POST failed [%d][%s]", res, httpResToStr(res));
            else
                logInfo("HTTPS POST succeeded [%d]\r\n%s", http.getHTTPResponseCode(), http_rx_buf);
            
            print_timer.reset();
            
        }

// SMS
        if (sms_timer.read_ms() > sms_interval_ms) {
            sms_timer.reset();
            logInfo("SMS Send Routine");
            printf("  In sms routine \r\n");
            if (radio_ok) {
                MbedJSONValue sms_json;
                string sms_str;

                sms_json["  edge Gravity: "] = dataY;
                sms_json[" level Gravity: "] = dataZ;

                sms_str = "SENSOR DATA:\n";
                sms_str += sms_json.serialize();

                logDebug("sending SMS to %s:\r\n%s", phone_number.c_str(), sms_str.c_str());
                if (dataZ<0.6f) {   //added
                Code ret = radio->sendSMS(phone_number, sms_str);
                if (ret != MTS_SUCCESS)
                    logError("sending SMS failed");
                    }     //added
            }
        }
    }
}

// init functions
bool init_mtsas()
{
    io = new MTSSerialFlowControl(RADIO_TX, RADIO_RX, RADIO_RTS, RADIO_CTS);
    if (! io)
        return false;

    io->baud(115200);
    radio = CellularFactory::create(io);
    if (! radio)
        return false;

    Code ret = radio->setApn(apn);
    if (ret != MTS_SUCCESS)
        return false;

    Transport::setTransport(radio);

    return true;
}

char* httpResToStr(HTTPResult res) {
    switch(res) {
        case HTTP_PROCESSING:
            return "HTTP_PROCESSING";
        case HTTP_PARSE:
            return "HTTP_PARSE";
        case HTTP_DNS:
            return "HTTP_DNS";
        case HTTP_PRTCL:
            return "HTTP_PRTCL";
        case HTTP_NOTFOUND:
            return "HTTP_NOTFOUND";
        case HTTP_REFUSED:
            return "HTTP_REFUSED";
        case HTTP_ERROR:
            return "HTTP_ERROR";
        case HTTP_TIMEOUT:
            return "HTTP_TIMEOUT";
        case HTTP_CONN:
            return "HTTP_CONN";
        case HTTP_CLOSED:
            return "HTTP_CLOSED";
        case HTTP_REDIRECT:
            return "HTTP_REDIRECT";
        case HTTP_OK:
            return "HTTP_OK";
        default:
            return "HTTP Result unknown";
    }
}
