/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "iBeaconService.h"
#include "UARTService.h"

/**
 * For this demo application, populate the beacon advertisement payload
 * with 2 AD structures: FLAG and MSD (manufacturer specific data).
 *
 * Reference:
 *  Bluetooth Core Specification 4.0 (Vol. 3), Part C, Section 11, 18
 */

BLEDevice ble;
UARTService *uart;

/**
 * The Beacon payload has the following composition:
 * 128-Bit / 16byte UUID = E2 0A 39 F4 73 F5 4B C4 A1 2F 17 D1 AD 07 A9 61
 * Major/Minor  = 0x1122 / 0x3344
 * Tx Power     = 0xC8 = 200, 2's compliment is 256-200 = (-56dB)
 *
 * Note: please remember to calibrate your beacons
 * TX Power for more accurate results.
 */
const uint8_t uuid[] = {0xE2, 0x0A, 0x39, 0xF4, 0x73, 0xF5, 0x4B, 0xC4,
                        0xA1, 0x2F, 0x17, 0xD1, 0xAD, 0x07, 0xA9, 0x62
                       };
uint16_t majorNumber = 1122;
uint16_t minorNumber = 3344;
uint16_t txPower = 0xC8;


const uint8_t b1_id   = 0x01;
const uint8_t b2_id   = 0x02;
const uint8_t b3_id   = 0x03;
const uint8_t b4_id   = 0x04;
const uint8_t b5_id   = 0x05;
const uint8_t b6_id   = 0x06;


const uint8_t bA1_id  = 0x31;
const uint8_t bA2_id  = 0x32;
const uint8_t bA3_id  = 0x33;
const uint8_t bA4_id  = 0x34;
const uint8_t bA5_id  = 0x35;
const uint8_t bA6_id  = 0x36;



PinName b1_p        = P0_4;
PinName b2_p        = P0_5;
PinName b3_p        = P0_11; //RXD
PinName b4_p        = P0_10; //CTX
PinName b5_p        = P0_9;  //TXD
PinName b6_p        = P0_8;  //RTS



//setup led
DigitalOut led1(P0_19);

//setup digital outs
DigitalOut b1(b1_p);
DigitalOut b2(b2_p);
DigitalOut b3(b3_p);
DigitalOut b4(b4_p);
DigitalOut b5(b5_p);
DigitalOut b6(b6_p);


void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason);
void onDataWritten(const GattCharacteristicWriteCBParams *params);
int flip(char value);

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    ble.startAdvertising();
}

void onDataWritten(const GattCharacteristicWriteCBParams *params)
{
    flip(params->data[0]);
}

int main(void)
{
    /* initialize stuff */
    led1 = 1;
    
    b1 = 1; // 0x01  flash hazards        in5       d4  p04
    b2 = 1; // 0x02  horn                 in6       d5  p05
    b3 = 1; // 0x03  headlights           in7       d6  p15
    b4 = 1; // 0x04  door locking  0.8    in2  in3  d2  cts
    b5 = 1; // 0x05  remote start  2x .25 in1       d1  txd
    b6 = 1; // 0x06  trip alarm    0.8    in4       d3  rts
    

    

    ble.init();
    
    ble.onDisconnection(disconnectionCallback);
    ble.onDataWritten(onDataWritten);
    
    
    uart = new UARTService(ble);    
    iBeaconService ibeacon(ble, uuid, majorNumber, minorNumber, txPower);

    ble.setAdvertisingInterval(1000); /* 1000ms. */
    ble.startAdvertising();

    while(1) {
        ble.waitForEvent(); // allows or low power operation
    }
}

void testAll(void)
{
    for(int i=0; i<2; i++)    
    {
        led1 = !led1;
        b1 = !b1;
        b2 = !b2;
        b3 = !b3;
        b4 = !b4;
        b5 = !b5;
        b6 = !b6;
        wait(1);
    }
}

void blinky(void)
{
    for(int i=0; i<15; i++)    
    {
        led1 = !led1;
        wait(.3);
    }
}

int flip(char value)
{
    int ret = 0;
    
    switch(value)
    {
        // 0x01  flash hazards        in5       d4  p4
        case b1_id  :
        case bA1_id :       led1 = !led1;
                            b1 = !b1;
                            break;
        // 0x02  horn                 in6       d5  p5
        case b2_id  :
        case bA2_id :       led1 = !led1;
                            b2 = !b2;
                            break;
        // 0x03  headlights           in7       d6  rxd
        case b3_id  :
        case bA3_id :       led1 = !led1;
                            b3 = !b3;
                            break;
        // 0x04  door locking  0.8    in2  in3  d2  cts
        case b4_id  :
        case bA4_id :       led1 = !led1;
                            b4 = !b4;
                            wait(0.8);
                            b4 = !b4;
                            break;
        // 0x05  remote start  2x .25 in1       d1  txd
        case b5_id  :
        case bA5_id :       led1 = !led1;
                            b5 = !b5;
                            wait(0.25);
                            b5 = !b5;
                            
                            b5 = !b5;
                            wait(0.25);
                            b5 = !b5;
                            break;

        // 0x06  trip alarm    0.8    in4       d3  rts
        case b6_id  :
        case bA6_id :       led1 = !led1;
                            b6 = !b6;
                            wait(0.8);
                            b6 = !b6;
                            break;
        
        default :           blinky();
                            ret = 1;
                            break;
    }
    return ret;
}