#ifndef BRADGAIT_H
#define BRADGAIT_H
#include "gait_generator.h"
#include "blend_generator.h"

//TODO: Add time_steps calculation in another file (App input should be a time in seconds)
//TODO: Add peak_time calculation in another file (App input should be a fraction)
//TODO: Add Feet Together Trajectories

/**
* Copyright (c) 2015
* All rights reserved.
* Bradley Perry quartic spline gait functions.
*
* @file Brad_poly_gait.h
* @author Bradley Perry
*
* @brief This contains the trajectories and gaits that define the Bradley Perry quartic spline gait.  This is dependent on blend_generator.h and gait_generator.h
*/

/** A struct that holds all the variables necessary to develop gaits.
*/
struct Brad_poly_gait_t {
    /** number of time steps per step
    */
    int time_steps;
    /** time of peak hip angle
    */
    float peak_time;
    /** stance start hip angle
    */
    float stance_start;
    /** stance end hip angle
    */
    float stance_end;
    /** max hip flexion angle
    */
    float max_angle;
    /** offset to hip angles during double stance for comfort
    */
    float doublestance_offset;
    /** maximum angle achieved during the first step
    */
    float max_fs_angle;
    /** the angle of both hips during standing
    */
    float standing_angle;
};

/** A class that defines the swing trajectory per Michael McKinley's design.
*/
class BradPolySwing: public TrajectoryGenerator
{
public:
    /** Construct the object by passing in a set of parameters.
    * @param p A set of parameters that defines the gait.
    */
    BradPolySwing (Brad_poly_gait_t& p);
    /** Basic constructor.
    */
    BradPolySwing();
    using TrajectoryGenerator::set;
    /** Pass in a set of gait parameters.
    * @param p A set of parameters that defines the gait.
    */
    void set(Brad_poly_gait_t& p);
    using TrajectoryGenerator::init;
    /** Initialize the linear blend by passing in a couple arguments.
    * @param start The position you want to start the linear blend from.
    * @param end The position you want to end the linear blend at.
    * @param time_steps The number of time steps needed for the linear blend.
    */
    virtual void init(float start, float end, int time_steps);
    using TrajectoryGenerator::calculate;
    /** Calculates the swing reference based on the current time.
    * @param time The time to calculate the trajectory on.
    * @param value Pointer to the variable that you're saving the trajectory at the specified time to.
    * @returns A flag that signals if there was an error.
    */
    virtual bool calculate(int time, float& value);
    /** Restart from the start of the trajectory.
    */
    virtual void restart();
private:
    /** The private struct to store all the parameters in.
    */
    Brad_poly_gait_t* _params;
    /** The linear blend object for blending.
    */
    LinearBlend _blend;

    /**Converts to normalized time*/
    float convert_to_tau(int time);

    /**Converts from the paramters to the matrix of coefficients*/
    void calculate_phi();

    /** The number of time steps spent blending.
    */
    int _blend_steps;

    /** A temporary variable that contains the polynomial coefficients */
    float _phi[5][2];

    /**A temporary variable that contains the change times */
    int _times[3];

    /** Which polynomial we have selected*/
    int _current_poly;

    /**Normalized time */
    float _tau;


};

/** A class that defines the stance trajectory per Michael McKinley's design.
*/

class BradPolyStance: public TrajectoryGenerator
{
public:
    /** Construct the object by passing in a set of parameters.
        * @param p A set of parameters that defines the gait.
        */
    BradPolyStance(Brad_poly_gait_t& p);
    /** Basic constructor.
    */
    BradPolyStance();
    using TrajectoryGenerator::set;
    /** Pass in a set of gait parameters.
    * @param p A set of parameters that defines the gait.
    */
    void set(Brad_poly_gait_t& p);
    using TrajectoryGenerator::init;
    /** Initialize the linear blend by passing in a couple arguments.
    * @param start The position you want to start the linear blend from.
    * @param end The position you want to end the linear blend at.
    * @param time_steps The number of time steps needed for the linear blend.
    */
    virtual void init(float start, float end, int time_steps);
    /** Calculates the stance reference based on the current time.
    * @param time The time to calculate the trajectory on.
    * @param value Pointer to the variable that you're saving the trajectory at the specified time to.
    * @returns A flag that signals if there was an error.
    */
    virtual bool calculate(int time, float& value);
    /** Restart from the start of the trajectory.
    */
    virtual void restart();
private:
    /** The private struct to store all the parameters in.
    */
    Brad_poly_gait_t* _params;
    /** The linear blend object for blending.
    */
    LinearBlend _blend;
    /**Converts to normalized time*/
    float convert_to_tau(int time);

    /**Converts from the paramters to the matrix of coefficients*/
    void calculate_phi();

    /** The number of time steps spent blending.
    */
    int _blend_steps;

    /** A temporary variable that contains the polynomial coefficients */
    float _phi[5][1];

    /**A temporary variable that contains the change times */
    int _times[2];

    /** Which polynomial we have selected*/
    int _current_poly;

    /**Normalized time */
    float _tau;

};

/** A class that defines the first step swing trajectory per Michael McKinley's design.
*/

class BradPolyFSSwing: public TrajectoryGenerator
{
public:
    /** Construct the object by passing in a set of parameters.
        * @param p A set of parameters that defines the gait.
        */
    BradPolyFSSwing(Brad_poly_gait_t& p);
    /** Basic constructor.
    */
    BradPolyFSSwing();
    using TrajectoryGenerator::set;
    /** Pass in a set of gait parameters.
    * @param p A set of parameters that defines the gait.
    */
    void set(Brad_poly_gait_t& p);
    using TrajectoryGenerator::init;
    /** Initialize the linear blend by passing in a couple arguments.
    * @param start The position you want to start the linear blend from.
    * @param end The position you want to end the linear blend at.
    * @param time_steps The number of time steps needed for the linear blend.
    */
    virtual void init(float start, float end, int time_steps);
    /** Calculates the first step swing reference based on the current time.
    * @param time The time to calculate the trajectory on.
    * @param value Pointer to the variable that you're saving the trajectory at the specified time to.
    * @returns A flag that signals if there was an error.
    */
    virtual bool calculate(int time, float& value);
private:
    /** The private struct to store all the parameters in.
    */
    Brad_poly_gait_t* _params;
    /** The linear blend object for blending.
    */
    LinearBlend _blend;
    /** The number of time steps spent blending.
    */
    int _blend_steps;
};

/** A class that defines the first step stance trajectory per Michael McKinley's design.
*/

class BradPolyFSStance: public TrajectoryGenerator
{
public:
    /** Construct the object by passing in a set of parameters.
        * @param p A set of parameters that defines the gait.
        */
    BradPolyFSStance(Brad_poly_gait_t& p);
    /** Basic constructor.
    */
    BradPolyFSStance();
    using TrajectoryGenerator::set;
    /** Pass in a set of gait parameters.
    * @param p A set of parameters that defines the gait.
    */
    void set(Brad_poly_gait_t& p);
    using TrajectoryGenerator::init;
    /** Initialize the linear blend by passing in a couple arguments.
    * @param start The position you want to start the linear blend from.
    * @param end The position you want to end the linear blend at.
    * @param time_steps The number of time steps needed for the linear blend.
    */
    virtual void init(float start, float end, int time_steps);
    /** Calculates the first step stance reference based on the current time.
    * @param time The time to calculate the trajectory on.
    * @param value Pointer to the variable that you're saving the trajectory at the specified time to.
    * @returns A flag that signals if there was an error.
    */
    virtual bool calculate(int time, float& value);
private:
    /** The private struct to store all the parameters in.
    */
    Brad_poly_gait_t* _params;
    /** The linear blend object for blending.
    */
    LinearBlend _blend;
    /** The number of time steps spent blending.
    */
    int _blend_steps;
};

#endif