#include "acc.h"

#define ACC_I2C_ADDR    (0x3a)

#define ACC_ADDR_XOUTL  0x00
#define ACC_ADDR_XOUTH  0x01
#define ACC_ADDR_YOUTL  0x02
#define ACC_ADDR_YOUTX  0x03
#define ACC_ADDR_ZOUTL  0x04
#define ACC_ADDR_ZOUTH  0x05
#define ACC_ADDR_XOUT8  0x06
#define ACC_ADDR_YOUT8  0x07
#define ACC_ADDR_ZOUT8  0x08
#define ACC_ADDR_STATUS 0x09
#define ACC_ADDR_DETSRC 0x0A
#define ACC_ADDR_TOUT   0x0B
#define ACC_ADDR_I2CAD  0x0D
#define ACC_ADDR_USRINF 0x0E
#define ACC_ADDR_WHOAMI 0x0F
#define ACC_ADDR_XOFFL  0x10
#define ACC_ADDR_XOFFH  0x11
#define ACC_ADDR_YOFFL  0x12
#define ACC_ADDR_YOFFH  0x13
#define ACC_ADDR_ZOFFL  0x14
#define ACC_ADDR_ZOFFH  0x15
#define ACC_ADDR_MCTL   0x16
#define ACC_ADDR_INTRST 0x17
#define ACC_ADDR_CTL1   0x18
#define ACC_ADDR_CTL2   0x19
#define ACC_ADDR_LDTH   0x1A
#define ACC_ADDR_PDTH   0x1B
#define ACC_ADDR_PW     0x1C
#define ACC_ADDR_LT     0x1D
#define ACC_ADDR_TW     0x1E

#define ACC_MCTL_MODE(m) ((m) << 0)
#define ACC_MCTL_GLVL(g) ((g) << 2)


#define ACC_STATUS_DRDY 0x01
#define ACC_STATUS_DOVR 0x02
#define ACC_STATUS_PERR 0x04


int ACC::getStatus(void) {
    char buf[1];

    buf[0] = ACC_ADDR_STATUS;
    _i2c.write(ACC_I2C_ADDR, buf, 1);
    _i2c.read(ACC_I2C_ADDR, buf, 1);
    
    return buf[0];
}

int ACC::getModeControl(void) {
    char buf[1];

    buf[0] = ACC_ADDR_MCTL;

    _i2c.write(ACC_I2C_ADDR, buf, 1);
    _i2c.read(ACC_I2C_ADDR, buf, 1);
    return buf[0];
}

void ACC::setModeControl(int mctl) {
    char buf[2];

    buf[0] = ACC_ADDR_MCTL;
    buf[1] = mctl;
    _i2c.write(ACC_I2C_ADDR, buf, 2);
}



/******************************************************************************
 *
 * Description:
 *    Initialize the ISL29003 Device
 *
 *****************************************************************************/
ACC::ACC (PinName sda, PinName scl) : _i2c(sda, scl) {

    _i2c.frequency(100000);
    /* set to measurement mode by default */
    setModeControl( (ACC_MCTL_MODE(ACC_MODE_MEASURE)
                     | ACC_MCTL_GLVL(ACC_RANGE_2G) ));

}

/******************************************************************************
 *
 * Description:
 *    Read accelerometer data
 *
 * Params:
 *   [out] x - read x value
 *   [out] y - read y value
 *   [out] z - read z value
 *
 *****************************************************************************/
void ACC::read (int *x, int *y, int *z) {
    char buf[1] = {0};

    /* wait for ready flag */
    while ((getStatus() & ACC_STATUS_DRDY) == 0);

    /*
     * Have experienced problems reading all registers
     * at once. Change to reading them one-by-one.
     */
    buf[0] = ACC_ADDR_XOUT8;

    _i2c.write(ACC_I2C_ADDR, buf, 1);
    _i2c.read(ACC_I2C_ADDR, buf, 1);

    *x = (int)(signed char)buf[0];

    buf[0] = ACC_ADDR_YOUT8;
    
    _i2c.write(ACC_I2C_ADDR, buf, 1);    
    _i2c.read(ACC_I2C_ADDR, buf, 1);

    *y = (int)(signed char)buf[0];

    buf[0] = ACC_ADDR_ZOUT8;

    _i2c.write(ACC_I2C_ADDR, buf, 1);
    _i2c.read(ACC_I2C_ADDR, buf, 1);

    *z = (int)(signed char)buf[0];
}

/******************************************************************************
 *
 * Description:
 *    Set the g-Range
 *
 * Params:
 *   [in] range - the g-Range
 *
 *****************************************************************************/
void ACC::setRange(acc_range_t range) {
    int mctl = 0;

    mctl = getModeControl();

    mctl &= ~(0x03 << 2);
    mctl |= ACC_MCTL_GLVL(range);

    setModeControl(mctl);
}

/******************************************************************************
 *
 * Description:
 *    Set sensor mode
 *
 * Params:
 *   [in] mode - the mode to set
 *
 *****************************************************************************/
void ACC::setMode(acc_mode_t mode) {
    int mctl = 0;

    mctl = getModeControl();

    mctl &= ~(0x03 << 0);
    mctl |= ACC_MCTL_MODE(mode);

    setModeControl(mctl);
}
