/*
 * osc.h
 *
 *  Created on: Mar 8, 2010
 *      Author: pehr
 *
 *       The functions to receive, parse and delegate OSC commands
 *       Based on osc_cpp.h in Make Controller repository
 *       This code is trimmed down to be UDP-only, no USB (code never had TCP)
 *       http://makingthings.com
 */
/*********************************************************************************

 Copyright 2006-2009 MakingThings

 Licensed under the Apache License,
 Version 2.0 (the "License"); you may not use this file except in compliance
 with the License. You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software distributed
 under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 CONDITIONS OF ANY KIND, either express or implied. See the License for
 the specific language governing permissions and limitations under the License.

*********************************************************************************/
#ifndef OSC_H_
#define OSC_H_

#include "ipv4/lwip/ip_addr.h"
#include "lwip/udp.h"

#ifndef UDP_BROADCAST_PORT
#define UDP_BROADCAST_PORT 5555
#endif

#include "error.h" //Make Controller error codes

#define OSC_SUBSYSTEM_COUNT  18

#define OSC_SCRATCH_SIZE    100

#define OSC_CHANNEL_COUNT    3
#define OSC_CHANNEL_UDP 0
#define OSC_CHANNEL_USB 1 //not implemented yet
#define OSC_CHANNEL_TCP 2 //not implemented yet


#define OSC_MAX_MESSAGE_IN   200
#define OSC_MAX_MESSAGE_OUT  600


typedef unsigned char uchar;

typedef struct OscChannel_
{
  char buffer[ OSC_MAX_MESSAGE_OUT ];
  char incoming[ OSC_MAX_MESSAGE_IN ];
  char* bufferPointer;
  int  bufferRemaining;
  int  messages;
  struct ip_addr * replyAddress; //where to send
  int replyPort;
  //A function pointer to the func used to send using this channel, set up in an init function
  int (*sendMessage)( char* packet, int length, struct ip_addr * replyAddress, int replyPort );
  int running;
}OscChannel;

// Top level functions
void Osc_SystemInit( struct udp_pcb * pcb );
void Osc_UdpInit( int channel, struct udp_pcb * pcb); //set up UDP channel

OscChannel * Osc_GetChannel(int  channel); //get ptr to the channel struct

int Osc_CreateMessage( int channel, char* address, char* format, ... );
int Osc_CreateMessageToBuf( char* bp, int* length, char* address, char* format, ... );
int Osc_SendPacket( int channel );
int Osc_ReceivePacket( int channel, char* packet, int length );
int Osc_RegisterSubsystem( const char *name,
                           int (*Subsystem_ReceiveMessage)( int channel, char* buffer, int length ),
                           int (*Subsystem_Poll)( int channel ) );

// Helpers for processing messages - they return error codes as spec'ed above
int Osc_IntReceiverHelper( int channel, char* message, int length,
                           char* subsystemName,
                           int (*propertySet)( int property, int value ),
                           int (*propertyGet)( int property ),
                           char* propertyNames[] );

int Osc_IndexIntReceiverHelper( int channel, char* message, int length,
                                int indexCount, bool (*validIndex)( int index ),char* subsystemName,
                                int (*propertySet)( int index, int property, int value ),
                                int (*propertyGet)( int index, int property ),
                                char* propertyNames[] );

int Osc_BlobReceiverHelper( int channel, char* message, int length,
                            char* subsystemName,
                            int (*blobPropertySet)( int property, uchar* buffer, int length ),
                            int (*blobPropertyGet)( int property, uchar* buffer, int size ),
                            char* blobPropertyNames[] );


int Osc_IndexBlobReceiverHelper( int channel, char* message, int length,
                                 int indexCount, char* subsystemName,
                                 int (*blobPropertySet)( int index, int property, uchar* buffer, int length ),
                                 int (*blobPropertyGet)( int index, int property, uchar* buffer, int size ),
                                 char* blobPropertyNames[] );

int Osc_GeneralReceiverHelper( int channel, char* message, int length,
                           char* subsystemName,
                           int (*propertySet)( int property, char* typedata, int channel ),
                           int (*propertyGet)( int property, int channel ),
                           char* propertyNames[] );

int Osc_SendError( int channel, char* subsystemName, int error );


// Functions needed to process messages
int Osc_SubsystemError( int channel, char* subsystem, char* string );
int Osc_PropertyLookup( char** properties, char* property );
char *Osc_FindDataTag( char* message, int length );
int Osc_ExtractData( char* buffer, char* format, ... );
int Osc_NumberMatch( int max, char* pattern, int* fancy );
int Osc_SetReplyAddress( int channel, struct ip_addr * replyAddress );
int Osc_SetReplyPort( int channel, int replyPort );

// Pattern Match Stuff
bool Osc_PatternMatch(const char *  pattern, const char * test);


#endif /* OSC_H_ */

