// OLED display using SSD1305 driver
// A library by *spark audio-visual

/* Copyright (c) 2011 Toby Harris, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
 
 /* rewritten to work with the SSD1306 driver JCY-MCU oled (from ebay). chris stevens 2013
 
 */
 

#ifndef SPK_OLED_SSD1305_h
#define SPK_OLED_SSD1305_h

#include "mbed.h"
#include <string>

#define bufferCount 1024
#define bufferWidth 128
#define pixelWidth 128
#define pixelHeight 64
#define pixInPage 8
#define pageCount 8



/** Display class for 128x64 OLEDs using the SSD1305 driver, connected via SPI
  *
  * Display ie. DENSITRON - DD-12864YO-3A
  *
  * This is a ground-up, minimal library. Further functionality as and when its needed or anybody wants to contribute.
  *
  * This library includes two processing sketches to create a font and full-screen image in the required byte representations.
  * Without creating your font and any images, all this library will do is blank the screen and draw horizontal lines. But at least you'll know its working!
  *
  * Terminology:
  * 'rows' are 8 pixel high rows across the display, 0 being the topmost and 7 the bottom.
  * 'lines' are single pixel lines, origin top left.
  *
*/
class SPKDisplay
{
public:
    /** Create a display object connected via SPI
     *
     * @param mosi  SPI MOSI
     * @param clk   SPI SCK
     * @param cs    Chip Select - a digital out pin
     * @param dc    Data/Command - a digital out pin
     * @param res   Reset - a digital out pin
     * @param debugSerial An optional serial object to log to
     */
    SPKDisplay(PinName mosi, PinName clk, PinName cs, PinName dc, PinName res, Serial *debugSerial = NULL);

    /** Font - Assign the ASCII value of the character at the start of the implemented range */
    const int *fontStartCharacter;

    /** Font - Assign the ASCII value of the character at the end of the implemented range */
    const int *fontEndCharacter;

    /** Font - Assign the font, an array of 8x8px characters
     *
     * @note The processing sketch spk_oled_fontByteMaker--processing takes characterCount*8px x 8px images and creates the code to declare the font array needed by this method
     */
    uint8_t const **fontCharacters;

    // set display for black on white
    void inverse();

// display a welcopme message encoded in tab0 in  spk_ole_ssd1305.h 
    void welcome();
    // output the entire font as a bitmap
    void fontdemo();
    
    //set display for white on black (normal state after reset)
    void normal();

    // power on / power off the display - this puts the display to sleep - but doesn turn off tha actuall power pin.
    void displayPower(int P);

    /** Completely clear the object's display representation */
    void clearBuffer();

    /** Clear a row of the object's display representation
     *
     * @param row The row to clear.
     */
    void clearBufferRow(int row);

    /** Replace the object\s display representation with the contents of image
     *
     * @param image An array of 1056 bytes representing an image.
     * @note The processing sketch spk_oled_screenByteMaker--processing takes 132x64 images and creates the code to declare such arrays
     */
    void imageToBuffer(const uint8_t* image);

    /** Draw a horizontal line in the object's display representation
     *
     * @param y The y position of the line to draw
     */
    void horizLineToBuffer(int y);


    /** Send the object's display representation to the OLED
     *
     * You can safely call this once per main loop, it will only transmit the buffer contents if there has been an update
     */
    void sendBuffer();
    
// send text to the buffer to write inthe tiny 5x7 font
void textToBuffer(char *message, int row);

private:
    SPI *spi;
    DigitalOut *cs;
    DigitalOut *dc;
    DigitalOut *res;

    Serial *debug;
    uint8_t buffer[bufferCount];

    bool bufferHasChanged;

    void setup();
};

#endif
