#include "mbed.h"
#include "servo_ppm.h"

//many things are "global" (not inside the class) because it's not easy to attach a class member function
// to a timer before instantiating the class.
//If you know a method to do that, let me know...

//to learn more about servos and PPM signals see this thread: http://www.rcgroups.com/forums/showthread.php?t=2040789

//global configuration var
int FRAME_PERIOD =          20000;  //us, time from beginning of a frame to beginning of next frame. It may increase to consider minimum frame interval
int MIN_FRAME_INTERVAL =    4000;   //us, also called "synchro blank time", see: http://www.rcgroups.com/forums/showthread.php?t=2040789
int MIN_SERVO_PULSE_DUR =   1000;   //us, decrease this value go to below -100% servo travel
int MAX_SERVO_PULSE_DUR =   2000;   //us, increase this value to go above +100% servo travel
int PPM_PULSE_DURATION =    400;    //us, between 200 and 500us

//global var
int servoPulseDuration[8];  //us, servo pulse duration (1000->2000), 8x2 bytes
int frameDuration;          //us, duration of one servo/PPM frame
int frameInterval;          //us, time from beginning of a frame to beginning of next frame.
int servoCnt;               //servo counter (0->7)
DigitalOut *ptrServo[8];    //servo output array
DigitalOut *ptrPpm;         //ppm output

//global timers                 
Timeout servoPulseTimeOut;
Timeout ppmPulseTimeOut;
Timeout frameIntervalTimeOut;
        
//global functions
void start(void);
void stop(void);
void endOfPpmPulse(void);
void endOfServoPulse(void);

//class
servo_ppm::servo_ppm( PinName PPM ,
            PinName S1 , PinName S2 , PinName S3 , PinName S4 ,
            PinName S5 , PinName S6 , PinName S7 , PinName S8 )
{
    //init ptr digital out
    if ( PPM != NC ) ptrPpm = new DigitalOut(PPM); else ptrPpm = NULL; 
    if ( S1 != NC ) ptrServo[0] = new DigitalOut(S1); else ptrServo[0] = NULL; 
    if ( S2 != NC ) ptrServo[1] = new DigitalOut(S2); else ptrServo[1] = NULL; 
    if ( S3 != NC ) ptrServo[2] = new DigitalOut(S3); else ptrServo[2] = NULL; 
    if ( S4 != NC ) ptrServo[3] = new DigitalOut(S4); else ptrServo[3] = NULL; 
    if ( S5 != NC ) ptrServo[4] = new DigitalOut(S5); else ptrServo[4] = NULL; 
    if ( S6 != NC ) ptrServo[5] = new DigitalOut(S6); else ptrServo[5] = NULL; 
    if ( S7 != NC ) ptrServo[6] = new DigitalOut(S7); else ptrServo[6] = NULL; 
    if ( S8 != NC ) ptrServo[7] = new DigitalOut(S8); else ptrServo[7] = NULL; 

    //init servo pulse duration to 1500us
    for (int i=0; i<8; i++) servoPulseDuration[i] = 1500;
}

////////////////////////
// endOfPpmPulse
////////////////////////

void endOfPpmPulse(void)
{
    assign( ptrPpm, 0);
}

////////////////////////
// endOfServoPulse
////////////////////////

void endOfServoPulse(void)
{
    //end of previous servo pulse
    assign( ptrServo[servoCnt], 0);
    
    //compute frame duration
    frameDuration += servoPulseDuration[servoCnt];
         servoCnt++;       
    //start next servo pulse if not end of frame
    if( servoCnt < 8 )
    {

        assign( ptrServo[servoCnt],1);
        servoPulseTimeOut.attach_us( &endOfServoPulse, servoPulseDuration[servoCnt] );
    }
    else
    {
        //end of frame: start frame interval timer
        frameInterval = FRAME_PERIOD - frameDuration;
        if ( frameInterval < MIN_FRAME_INTERVAL ) frameInterval = MIN_FRAME_INTERVAL;
        frameIntervalTimeOut.attach_us( &start, frameInterval );
    }
        
    //start ppm pulse and timer end of pulse ppm
    assign( ptrPpm, 1);
    ppmPulseTimeOut.attach_us( &endOfPpmPulse, 400 ); //400us
}

////////////////////////
// start
////////////////////////

void start(void)
{
    //init frame duration
    frameDuration = 0;
    
    //init servo counter
    servoCnt = 0;
    
    //start pulse servo 0 and timer end of pulse servo 0
    assign( ptrServo[0], 1);
    servoPulseTimeOut.attach_us( &endOfServoPulse, servoPulseDuration[servoCnt] );
   
    //start ppm pulse and timer end of pulse ppm
    assign( ptrPpm, 1);
    ppmPulseTimeOut.attach_us( &endOfPpmPulse, PPM_PULSE_DURATION );
}

////////////////////////
// stop()
////////////////////////

void stop(void)
{
    //stop all timers
    servoPulseTimeOut.detach();
    ppmPulseTimeOut.detach();
    frameIntervalTimeOut.detach();
    
    //reset output
    for (int i=0; i<8; i++) assign( ptrServo[i], 0 );
    assign( ptrPpm, 0);
}

///////////////////////////////
// startServoPpmOutput()
///////////////////////////////
void servo_ppm::startServoPpmOutput(void)
{
    start();
}
///////////////////////////////
// stopServoPpmOutput()
///////////////////////////////
void servo_ppm::stopServoPpmOutput(void)
{
    stop();
}

///////////////////////////////
// setServoPulseDuration_us()
///////////////////////////////
void servo_ppm::setServoPulseDuration_us(int numServo, int pulseDuration)
{
    if ((numServo>0) && (numServo<9))
    {
        if (pulseDuration > MAX_SERVO_PULSE_DUR)
            servoPulseDuration[numServo-1] = MAX_SERVO_PULSE_DUR;
        else
        {
            if (pulseDuration < MIN_SERVO_PULSE_DUR)
                servoPulseDuration[numServo-1] = MIN_SERVO_PULSE_DUR;
            else  
                servoPulseDuration[numServo-1] = pulseDuration;
        }
    }
}

///////////////////////////////
// setFramePeriod_us()
///////////////////////////////
void setFramePeriod_us( int val )
{
    FRAME_PERIOD = val;
}

///////////////////////////////
// setMinFrameInterval_us()
///////////////////////////////
void setMinFrameInterval_us( int val )
{
    MIN_FRAME_INTERVAL = val;
}

///////////////////////////////
// setMinServoPulseDuration_us()
///////////////////////////////
void setMinServoPulseDuration_us( int val )
{
    MIN_SERVO_PULSE_DUR = val;
}

///////////////////////////////
// setMaxServoPulseDuration_us()
///////////////////////////////
void setMaxServoPulseDuration_us( int val )
{
    MAX_SERVO_PULSE_DUR = val;
}

///////////////////////////////
// setPpmPulseDuration_us()
///////////////////////////////
void setPpmPulseDuration_us( int val )
{
    PPM_PULSE_DURATION = val;
}
