////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Flysky RC receiver 8 x servos PWM + PPM outputs
//
// tested with FRDM-KL05Z and A7105SY module 
// tested with custom board with KL05ZLVC4 chip and A7105SY module
//
// In order to program an external KL05Z32VLC4, I take a FRDM-KL25Z board (don't take FRDM-KL05Z board, there's a bug on layout!),
// cut wire under JP11 connector to disconnect onboard KL25Z chip, do a special cable (see FRDM-KL25Z schematic, J6 SWD CONNECTOR, 
// you just need to solder 5 wires: 3.3V, GND, SWD_DIO, SWD_CLK, RST_TGT) to link your custom KL05Z32VLC4 board, and program FRDM-KL25Z
// board with an FRDM-KL05Z firmware and it works fine as a cheap programmer interface!
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//compiler option
#define DEBUG               1   //display status through serial
#define STORE_TX_ID_KLxx    1   //store in flash memory previous TX ID in KLxx flash memory
                                //not necessary, but you'll need to bind transmitter each time you turn off your receiver 

//include
#include "mbed.h"
#include "A7105_FLYSKY_RX.h"    //Flysky protocol dialog with A7105 2.4Ghz module
#include "servo_ppm.h"          //servo PWM and PPM outputs
#if STORE_TX_ID_KLxx
#include "FreescaleIAP.h"       //to store TX ID in KLxx flash memory, only works for freescale KLxx
#endif                          //you need to use suitable lib for other arm family,
                                //and modify code linked to "#if STORE_TX_ID_KLxx" below...

//A7105 <-> KL05Z pin assignment
#define PIN_SDIO    PTA7    //SPI mosi
#define PIN_GIO1    PTA6    //SPI miso
#define PIN_SCK     PTB0    //SPI sck  
#define PIN_CS      PTA10   //CS
#define PIN_GIO2    PTA11   //wait rx

//servo <-> KL05Z pin assignment
#define PIN_SERVO1  PTB5
#define PIN_SERVO2  PTA12
#define PIN_SERVO3  PTB6
#define PIN_SERVO4  PTB7
#define PIN_SERVO5  PTB11
#define PIN_SERVO6  PTA5
#define PIN_SERVO7  PTA9
#define PIN_SERVO8  PTA8

//PPM <-> KL05Z pin assignment
#define PIN_PPM     PTB3

//serial debug
#if DEBUG
Serial pc(USBTX, USBRX);
#endif

//A7105 flysky RX
A7105_flysky_RX rx(PIN_SDIO, PIN_GIO1, PIN_SCK, PIN_CS, PIN_GIO2, LED_GREEN); //mosi, miso, sck, cs, wait_rx, status_led)

//servo and ppm outputs
//servo_ppm servoPpmOut( PIN_PPM, PIN_SERVO1, PIN_SERVO2, PIN_SERVO3,  PIN_SERVO4,  PIN_SERVO5,  PIN_SERVO6,  PIN_SERVO7,  PIN_SERVO8);
servo_ppm servoPpmOut( NC, PIN_SERVO1, PIN_SERVO2, NC,  PIN_SERVO4,  PIN_SERVO5,  PIN_SERVO6,  PIN_SERVO7,  PIN_SERVO8);
 
/////////////
// M A I N
/////////////

int main()
{      
    int i;
    bool ok;
       
    #if DEBUG
    //say hello!
    pc.printf("\n\rHello!");
    #endif
    
    //start servo/ppm frame
    servoPpmOut.startServoPpmOutput();
     
    //init A7105
    ok = rx.init();
    if (ok)
    {
        //bind TX
        ok = rx.bind(); //if bind failed, TX Id take "default_tx_id" value found in A7105_FLYSKY_RX.h
        
        #if STORE_TX_ID_KLxx
            //if bind ok, stores TX ID in flash memory
            int address = flash_size() - SECTOR_SIZE;           //Write in last sector
            if (ok)
            {
                erase_sector(address);
                #if DEBUG
                    printf("\n\r bind ok: write TX ID in Flash: %X,%X,%X,%X", rx.txId[0], rx.txId[1], rx.txId[2], rx.txId[3]);
                #endif
                program_flash(address, (char*)rx.txId, 4); //4 bytes
            }
            //if bind not ok, get previous bind TX ID, stored in flash memory
            else
            {
                char *data = (char*)address;
                rx.txId[0] = data[0]; rx.txId[1] = data[1]; rx.txId[2] = data[2]; rx.txId[3] = data[3];
                #if DEBUG
                    printf("\n\r bind failed, TX ID readden in Flash: %X,%X,%X,%X", rx.txId[0], rx.txId[1], rx.txId[2], rx.txId[3]);
                #endif
            }
        #else
            #if DEBUG
            //display bind status
            if (ok) printf("\r\nbind ok, TX ID  : %X,%X,%X,%X", rx.txId[0], rx.txId[1], rx.txId[2], rx.txId[3]);
            else printf("\r\nbind failed, take default TX ID  : %X,%X,%X,%X", rx.txId[0], rx.txId[1], rx.txId[2], rx.txId[3]);
            #endif
        #endif                    
        //infinite loop, receive packet and update servos
        int cptError = 0;
        while (1)
        {   
            //wait packet (1.46ms) or timeout(1.5ms) since last call to this function
            switch( rx.waitPacket() )
            {
                case NO_ERROR: //update servo position
                    for (i=0; i<6; i++) servoPpmOut.setServoPulseDuration_us( i+1, rx.servoPulseDur[i] ); break;
                case TIME_OUT_ERROR:
                    printf("\r\n%d: time out error", cptError); cptError++; break;
                case VALIDITY_ERROR:
                    printf("\r\n%d: validity error", cptError); cptError++; break;
                case TX_ID_ERROR:
                    printf("\r\n%d: tx id error", cptError); cptError++; break;
            }           
        }
    }
    #if DEBUG
    else
    {
         printf("\r\ninit failed !!!");
    }
    #endif
}
