#include "mbed.h"
#include "rtos.h"
#include "PID.h"

//Photointerrupter input pins
#define I1pin D2
#define I2pin D11
#define I3pin D12

//Incremental encoder input pins
#define CHA   D7
#define CHB   D8

//Motor Drive output pins   //Mask in output byte
#define L1Lpin D4           //0x01
#define L1Hpin D5           //0x02
#define L2Lpin D3           //0x04
#define L2Hpin D6           //0x08
#define L3Lpin D9           //0x10
#define L3Hpin D10          //0x20

//Mapping from sequential drive states to motor phase outputs
/*
   State   L1  L2  L3
   0       H   -   L
   1       -   H   L
   2       L   H   -
   3       L   -   H
   4       -   L   H
   5       H   L   -
   6       -   -   -
   7       -   -   -
   */


//https://developer.mbed.org/users/mbed_official/code/mbed-dev/file/default/targets/TARGET_STM/TARGET_STM32F3/TARGET_STM32F303x8/TARGET_NUCLEO_F303K8/PinNames.h

//////////////////// Constant Variables /////////////////////////

const int8_t driveTable[] = {0x12,0x18,0x09,0x21,0x24,0x06,0x00,0x00};
const int8_t stateMap[] = {0x07,0x05,0x03,0x04,0x01,0x00,0x02,0x07};
const int buffer_size = 255;
const double Rate = 0.5;
const double Propor = 9;
const double Interg = 25;
const double Deriv = 0;
const double Dist_Rate = 0.5;
const double Dist_Propor = 1000;
const double Dist_Interg = 50;
const double Dist_Deriv = 15;
enum NOTES_NAMES {  A, A_sharp, A_flat,
                    B, B_sharp, B_flat,
                    C, C_sharp, C_flat,
                    D, D_sharp, D_flat,
                    E, E_sharp, E_flat,
                    F, F_sharp, F_flat,
                    G, G_sharp, G_flat
                 };

//////////////////// Global Variables /////////////////////////

//Variable                                                  |   Sensors   ||   Distance   ||    Speed    ||     Song    ||   Command   ||    Tx/Rx    ||    Main    ||
//--------------------------------------------------------------------------------------------------------------------------------------------------------------------
//                                                          |Read | Write || Read | Write ||Read | Write ||Read | Write ||Read | Write ||Read | Write ||Read | Write||
RawSerial pc(SERIAL_TX, SERIAL_RX); //                      |     |       ||      |       ||     |       ||     |       ||     |       || X   |   X   ||     |   X  ||                         
DigitalOut led1(LED1);              //                      |     |       ||      |       ||     |       ||     |       ||     |       ||     |       ||     |      ||                       |                                                                   
DigitalOut led2(LED2);              //                      |     |       ||      |       ||     |       ||     |       ||     |       ||     |       ||     |      || 
DigitalOut led3(LED3);              //                      |     |       ||      |       ||     |       ||     |       ||     |   X   ||     |       ||     |      ||
PwmOut L1L(L1Lpin);                 //                      |     |       ||      |   X   ||     |       ||     |   X   ||     |   X   ||     |       ||     |   X  ||
DigitalOut L1H(L1Hpin);             //                      |     |       ||      |   X   ||     |       ||     |   X   ||     |   X   ||     |       ||     |   X  ||
PwmOut L2L(L2Lpin);                 //                      |     |       ||      |   X   ||     |       ||     |   X   ||     |   X   ||     |       ||     |   X  ||                          
DigitalOut L2H(L2Hpin);             //                      |     |       ||      |   X   ||     |       ||     |   X   ||     |   X   ||     |       ||     |   X  ||
PwmOut L3L(L3Lpin);                 //                      |     |       ||      |   X   ||     |       ||     |   X   ||     |   X   ||     |       ||     |   X  ||
DigitalOut L3H(L3Hpin);             //                      |     |       ||      |   X   ||     |       ||     |   X   ||     |   X   ||     |       ||     |   X  ||
volatile int8_t lead = -2;          //                      |  X  |       ||      |       ||  X  |       ||     |       ||     |   X   ||     |       ||     |      ||
volatile int8_t ofstate =4;         //                      |  X  |       ||      |       ||     |       ||     |       ||     |       ||     |       ||     |   X  ||
float pwm_duty = 1;                 //                      |  X  |   X   ||      |       ||     |   X   ||     |   X   ||     |       ||     |       ||     |      ||
char tx_buffer[buffer_size+1];      //                      |     |       ||      |       ||     |       ||     |       ||     |       ||  X  |   X   ||     |      ||                          
char rx_buffer[buffer_size+1];      //                      |     |       ||      |       ||     |       ||     |       ||     |       ||  X  |   X   ||     |      ||
volatile int tx_in=0;               //                      |     |       ||      |       ||     |       ||     |       ||     |       ||  X  |   X   ||     |      ||
volatile int tx_out=0;              //                      |     |       ||      |       ||     |       ||     |       ||     |       ||  X  |   X   ||     |      ||
volatile int rx_in=0;               //                      |     |       ||      |       ||     |       ||     |       ||     |       ||  X  |   X   ||     |      ||
volatile int rx_out=0;              //                      |     |       ||      |       ||     |       ||     |       ||     |       ||  X  |   X   ||     |      ||
char tx_line[80];                   //                      |     |   X   ||      |       ||     |       ||     |   X   ||     |   X   ||  X  |   X   ||     |   X  ||
char rx_line[80];                   //                      |     |       ||      |       ||     |       ||     |       ||  X  |   X   ||  X  |   X   ||     |      ||
bool song = false;                  //                      |     |       ||      |       ||  X  |       ||     |       ||     |   X   ||     |       ||     |      ||
volatile int turns = 0;             //                      |     |   X   ||   X  |   X   ||     |       ||     |       ||     |       ||     |       ||  X  |   X  ||
volatile int count = 0;             //                      |     |   X   ||   X  |   X   ||     |       ||     |       ||     |       ||     |       ||  X  |   X  ||
int countold = 0;                   //                      |     |       ||      |       ||     |       ||     |       ||     |       ||     |       ||  X  |   X  ||                          
int Direction_I = 1;                //                      |     |       ||      |       ||     |       ||     |       ||     |   X   ||     |       ||     |      ||
PID constspeed(Propor,Interg,Deriv,Rate); //                |     |       ||      |       ||     |   X   ||     |       ||     |       ||     |       ||     |   X  ||
PID SetDist(Dist_Propor,Dist_Interg,Dist_Deriv, Dist_Rate);//|    |       ||      |   X   ||     |       ||     |       ||     |   X   ||     |       ||     |   X  ||
double a;                           //                      |     |       ||      |   X   ||     |       ||     |       ||     |       ||     |       ||  X  |      ||
int current_distance;               //                      |     |       ||      |       ||     |       ||     |       ||     |       ||     |       ||     |      ||
volatile int target_distance;       //                      |     |       ||   X  |       ||     |       ||     |       ||     |   X   ||     |       ||     |      ||
volatile double rotation_time = 0;  //                      |     |       ||      |       ||  X  |       ||     |       ||     |       ||     |       ||  X  |  X   ||
volatile float num = 0.0;           //                      |     |       ||      |       ||     |       ||     |       ||     |       ||     |       ||     |      ||
volatile bool vel = false;          //                      |     |       ||      |       ||     |       ||     |       ||  X  |   X   ||     |       ||     |      ||
volatile bool rev = false;          //                      |     |       ||      |       ||     |       ||     |       ||  X  |   X   ||     |       ||     |      ||
volatile double n_rev = 0;          //                      |     |       ||      |       ||     |       ||     |       ||  X  |   X   ||     |       ||     |      ||
volatile double n_vel = 0;          //                      |     |       ||      |       ||     |       ||     |       ||  X  |   X   ||     |       ||     |      ||
bool point = false;                 //                      |     |       ||      |       ||     |       ||     |       ||  X  |   X   ||     |       ||     |      ||
bool minus = false;                 //                      |     |       ||      |       ||     |       ||     |       ||  X  |   X   ||     |       ||     |      ||
volatile bool output_main = false;  //                      |     |       ||      |   X   ||     |       ||     |   X   ||     |   X   ||     |       ||  X  |      ||
NOTES_NAMES notes [15];             //                      |     |       ||      |       ||     |       ||  X  |       ||     |   X   ||     |       ||     |      ||
int durations [15];                 //                      |     |       ||      |       ||     |       ||  X  |       ||     |   X   ||     |       ||     |      ||
int notes_length;                   //                      |     |       ||      |       ||     |       ||  X  |       ||     |   X   ||     |       ||     |      ||
bool run_dist = false;              //                      |     |       ||  X   |       ||     |       ||     |       ||     |   X   ||     |       ||     |      || 
bool run_song = false;              //                      |     |       ||      |       ||     |       ||  X  |       ||     |   X   ||     |       ||     |      ||
Mutex line_mutex;                   //                      |  X  |  X    ||  X   |   X   ||     |       ||   X |    X  ||   X |   X   ||   X |   X   ||   X | X    ||
Thread set_commands(osPriorityHigh, 1024, NULL);//          |     |       ||      |       ||     |       ||     |       ||     |       ||     |       ||     |      ||
Thread pid_thread(osPriorityNormal, 340, NULL);//           |     |       ||      |       ||     |       ||     |       ||     |       ||     |       ||     |      ||
Thread song_thread(osPriorityNormal, 512, NULL);//          |     |       ||      |       ||     |       ||     |       ||     |       ||     |       ||     |      ||
Thread distance_thread(osPriorityNormal, 340, NULL);//      |     |       ||      |       ||     |       ||     |       ||     |       ||     |       ||     |      ||


//////////////////// Function Prototypes /////////////////////////

void motorOut(int8_t driveState);
void distance();
void set_speed(double new_rotation);
void set_commands_func();
void song_func();
void send_line();
void read_line();
void Rx_interrupt();
void Tx_interrupt();
void distance();
void feedback_setup();
void pid_thread_func();

//Class to update through interrupts the field posistion, start it and stop it
class Position
{
public:
    Position(PinName pin_sensor1, PinName pin_sensor2, PinName pin_sensor3) :
        running(false),
        sensor1(pin_sensor1),
        sensor2(pin_sensor2),
        sensor3(pin_sensor3) {
        sensor1.rise(callback(this, &Position::sensor1_rise));
        sensor2.rise(callback(this, &Position::sensor2_rise));
        sensor3.rise(callback(this, &Position::sensor3_rise));
        sensor1.fall(callback(this, &Position::sensor1_fall));
        sensor2.fall(callback(this, &Position::sensor2_fall));
        sensor3.fall(callback(this, &Position::sensor3_fall));
        
        //Sensor1 interrupt always enabled to allow speed measurement after stop
        sensor2.disable_irq();
        sensor3.disable_irq();
    }

    //Read motor origin to set ofstate
    int8_t motor_home() {
        pwm_duty = 1;
        motorOut(0);
        wait(5);
        led3=1;
        int8_t home_state=stateMap[sensor1 + 2*sensor2 + 4*sensor3];
        motorOut(3);
        return home_state;
    }

    //Interrupts on sensor status changes(rise and fall) update to the
    //next state acording to lead
    inline void sensor1_rise() {
        count++;
        if(sensor2 == 0)turns++;
        if(sensor2 == 1)turns--;
        if (running) {
            if(lead == -2) motorOut((10-ofstate)%6);
            else motorOut((12-ofstate)%6);
        }
    }

    inline void sensor2_rise() {
        if(lead == -2) motorOut((14-ofstate)%6);
        else motorOut((10-ofstate)%6);
    }

    inline void sensor3_rise() {
        if(lead == -2) motorOut((12-ofstate)%6);
        else motorOut((14-ofstate)%6);
    }

    inline void sensor1_fall() {
        if(running) {
            if(lead == -2) motorOut((13-ofstate)%6);
            else motorOut((15-ofstate)%6);
        }
    }

    inline void sensor2_fall() {
        if(lead == -2) motorOut((11-ofstate)%6);
        else motorOut((13-ofstate)%6);
    }

    inline void sensor3_fall() {
        if(lead == -2) motorOut((15-ofstate)%6);
        else motorOut((11-ofstate)%6);
    }
    
    inline void stop() {
        //Running prevents interrupts on sensor 1 from updating field
        running = false;
        sensor2.disable_irq();
        sensor3.disable_irq();
        int8_t old_state = stateMap[sensor1 + 2*sensor2 + 4*sensor3];
        //Set field to current state and wait for stop
        motorOut((old_state-ofstate+6)%6);
        wait(6);
        pwm_duty = 0;
    }

    inline void start() {
        running = true;
        sensor2.enable_irq();
        sensor3.enable_irq();
        int8_t state = stateMap[sensor1 + 2*sensor2 + 4*sensor3];
        pwm_duty = 1;
        motorOut((state-ofstate+lead+6)%6);
    }

private:
    bool running;
    
    //https://docs.mbed.com/docs/mbed-os-api-reference/en/5.1/APIs/io/InterruptIn/
    InterruptIn sensor1;
    InterruptIn sensor2;
    InterruptIn sensor3;
};
Position sensors(I1pin, I2pin, I3pin);



////////////////////////////// Main ////////////////////////////
int main()
{
    //Serial pc communication through our own interrupts based on code from:
    //https://developer.mbed.org/cookbook/Serial-Interrupts
    pc.baud(9600);
    pc.attach(&Rx_interrupt, Serial::RxIrq);
    pc.attach(&Tx_interrupt, Serial::TxIrq);

    //Output to pc by writing to buffer and triggering comunications,
    //Lock to prevent race condition by threads
    line_mutex.lock();
    sprintf(tx_line, "Hello, aligning\r\n");
    send_line();
    line_mutex.unlock();


    //Run the motor synchronisation
    ofstate = sensors.motor_home();
    line_mutex.lock();
    sprintf(tx_line, "Rotor origin: %i\r\n",ofstate);
    send_line();
    line_mutex.unlock();

    //Frequency of main debug output
    int count2 =0;

    //Set pmw period for speed control
    L1L.period_us(10);
    L2L.period_us(10);
    L3L.period_us(10);

    feedback_setup();

    count = 0;
    turns = 0;

    //Start all threads, all will wait for command
    set_commands.start(set_commands_func);
    pid_thread.start(pid_thread_func);
    distance_thread.start(distance);
    song_thread.start(song_func);

    line_mutex.lock();
    sprintf(tx_line, "Waiting for new command\r\n");
    send_line();
    line_mutex.unlock();

    Timer t;
    t.start();

    while (1) {
        wait(Rate);

        rotation_time = (turns-countold)/t.read();//calculating rotation from I1
        countold = turns;
        t.reset();

        if (output_main && count2>0) { //report every so often as not to slow down loop
            line_mutex.lock();
            sprintf(tx_line, "Hz: %.2f PWM: %.2f Turns: %d Dist_pid: %.2f\r\n", rotation_time , pwm_duty , turns, a);
            send_line();
            line_mutex.unlock();
            count2 = 0;
        }
        count2++;

    } 
} 


/////////////////////// Function Declarations ///////////////////////////////

//Set a given drive state
void motorOut(int8_t driveState)
{
    //Lookup the output byte from the drive state.
    int8_t driveOut = driveTable[driveState & 0x07];

    //Turn off first
    if (~driveOut & 0x01) L1L.write(0);
    if (~driveOut & 0x02) L1H = 1;
    if (~driveOut & 0x04) L2L.write(0);
    if (~driveOut & 0x08) L2H = 1;
    if (~driveOut & 0x10) L3L.write(0);
    if (~driveOut & 0x20) L3H = 1;

    //Then turn on
    if (driveOut & 0x01) L1L.write(pwm_duty);
    if (driveOut & 0x02) L1H = 0;
    if (driveOut & 0x04) L2L.write(pwm_duty);
    if (driveOut & 0x08) L2H = 0;
    if (driveOut & 0x10) L3L.write(pwm_duty);
    if (driveOut & 0x20) L3H = 0;
}

// API for PID from https://developer.mbed.org/cookbook/PID
void feedback_setup()
{
    //Feedback for speed controller
    constspeed.setMode(1);
    constspeed.setOutputLimits(0.00, 1.0);
    constspeed.setInputLimits(-100.0, 100.0);

    //Feedback for distance controller
    SetDist.setMode(AUTO_MODE);
    SetDist.setOutputLimits(-50.0,50.0);
    SetDist.setInputLimits(-1000000,1000000);
    SetDist.setBias(0.0);
}

//THREAD: Always update target PMW output required for set speed
void pid_thread_func()
{   
    while(true) {
        if(lead == -2)constspeed.setProcessValue(rotation_time);
        else constspeed.setProcessValue(-1.0*rotation_time);
        if(song == false){
            pwm_duty = constspeed.compute();
        }
        Thread::wait(Rate*1000);// Rate is in seconds from the PID
    }
}

//Called by set_commands to set required speed and derive lead
void set_speed(double new_rotation)
{
    if(new_rotation >0) {
        new_rotation = new_rotation*1.0;
        lead = -2;
        Direction_I = 1;

    } else {
        lead = 2;
        new_rotation = new_rotation*-1.0;
        Direction_I = -1;
    }
    constspeed.setSetPoint(new_rotation);
}

//THREAD: Triggered by a new line character read, 
//        Parses the command and provides debug output
//        Blocks all threads from previous commands and unblocks
//        required new threads
void set_commands_func()
{
    //Length of the command guaranteed a fixed maximum length by regex smaller than this
    char command[32];
    while(true) {
        //Wait for command available
        Thread::signal_wait(0x1);
        
        read_line();
        sscanf(rx_line,"%s",command);
        line_mutex.lock();
        sprintf(tx_line, "Stopping (4s) to process new command: %s\r\n", command);
        send_line();
        line_mutex.unlock();
        
        run_dist = false;
        run_song = false;
        turns = 0;
        count = 0;
        sensors.stop();
        L1L.period_us(10);
        L2L.period_us(10);
        L3L.period_us(10);
        
        int length_command = 0;
        while(command[length_command] != 0) {
            line_mutex.lock();
            sprintf(tx_line, "     char: %c %i\r\n", command[length_command], command[length_command]);
            send_line();
            line_mutex.unlock();
            
            length_command++;
        }
        
        line_mutex.lock();
        sprintf(tx_line, "length: %i\r\n", length_command);
        send_line();
        line_mutex.unlock();


        vel = false; //command is stored into these variables
        rev = false; //
        n_rev = 0; //
        n_vel = 0; //
        point = false;
        minus = false;
        song = false;
        int i = 0;
        if(command[i]=='T') {
            song = true;
            i++;
            int j = 0;
            while (i < length_command) {
                bool sharp = false;
                bool flat = false;
                char note = command[i];
                i++;
                if (command[i]=='#') {
                    sharp=true;
                    i++;
                }
                if (command[i]=='^') {
                    flat=true;
                    i++;
                }
                switch (note) {
                    case 'A':
                        notes[j] = sharp ? A_sharp : (flat ? A_flat : A);
                        break;
                    case 'B':
                        notes[j] = sharp ? B_sharp : (flat ? B_flat : B);
                        break;
                    case 'C':
                        notes[j] = sharp ? C_sharp : (flat ? C_flat : C);
                        break;
                    case 'D':
                        notes[j] = sharp ? D_sharp : (flat ? D_flat : D);
                        break;
                    case 'E':
                        notes[j] = sharp ? E_sharp : (flat ? E_flat : E);
                        break;
                    case 'F':
                        notes[j] = sharp ? F_sharp : (flat ? F_flat : F);
                        break;
                    case 'G':
                        notes[j] = sharp ? G_sharp : (flat ? G_flat : G);
                        break;
                    default:
                        line_mutex.lock();
                        sprintf(tx_line, "Invalid note %c with int val %i\r\n ", note, note);
                        send_line();
                        line_mutex.unlock();
                }
                durations[j]=command[i]-48;
                sprintf(tx_line, "Note %i\r\n ", note);
                send_line();
                sprintf(tx_line, "Val %i\r\n ", notes[j]);
                send_line();
                i++;
                j++;
            }
            notes_length = j;
        } else {
            if(command[i]=='R') { 
                rev = true;
                i++;
                if (command[i]=='-') {
                    minus = true;   
                    i++;
                }
                while (point==false && i<length_command && command[i] != 'V') { //nr of digits in the command
                    if (command[i]!='.') {
                        n_rev = (n_rev)*10 + (command[i]-48);
                        i++;
                    }
                    if (command[i]=='.') {
                        point=true;
                        float dec_place = 0.1;
                        while (i < length_command && command[i] != 'V') {
                            n_rev = n_rev + (command[i]-48)*dec_place ;
                            dec_place = dec_place*0.1;
                            i++;
                        }
                    }
                }
                if(minus)n_rev = -n_rev;
            } 
            //If there is still command left to parse -> V command as well
            if (i < length_command) { 
                if(command[i]=='V') {  
                    vel = true;
                    i++;
                    minus = false;
                    if (command[i]=='-') {
                        minus = true;  
                        i++;
                    }
                    point = false;
                    while (point==false && i<length_command) { 
                        if (command[i]!='.') {
                            n_vel = (n_vel)*10 + (command[i]-48);
                            i++;
                        }
                        if (command[i]=='.') {
                            point=true;
                            float dec_place = 0.1;
                            for (int j = (i+1); j<length_command; j++) {
                                n_vel = n_vel + (command[i]-48)*dec_place ;
                                dec_place = dec_place*0.1;
                            }
                        }
                    }
                    if(minus)n_vel = -n_vel;
                } else {
                    line_mutex.lock();
                    sprintf(tx_line, "Invalid command syntax\r\n");
                    send_line();
                    line_mutex.unlock();
                }
            }
        }
        if (song) {
            sprintf(tx_line, "Read song command\r\n");
            send_line();
            for (int a = 0; a < notes_length; a++) {
                line_mutex.lock();
                sprintf(tx_line, "Note: %i Duration: %i\r\n", notes[a], durations[a]);
                send_line();
                line_mutex.unlock();
            }
            run_song = true;
            sensors.start();
            song_thread.signal_set(0x3);
        } else {
            if(rev) {
                line_mutex.lock();
                sprintf(tx_line, "Setting Revolutions %f\r\n ", n_rev);
                send_line();
                line_mutex.unlock();
                target_distance = n_rev;
                if(vel)set_speed(n_vel);
                else set_speed(10); 
                run_dist = true;
                sensors.start();
                distance_thread.signal_set(0x2);
            }
            if(vel) {
                line_mutex.lock();
                sprintf(tx_line, "Setting Velocity %f\r\n ", n_vel);
                send_line();
                line_mutex.unlock();
                if(!rev) {
                    set_speed(n_vel);
                    sensors.start();
                }
            }
        }
        output_main = true;
    }
}

//THREAD: Accelerate motor and then set PMW period to play notes, stop motor on end
void song_func ()
{
    /* Note frequenies (hz) - need the period (us) = 10^6/frequency
    const double freq[21] = {440, 466.16, 415.30,  //A4, A_sharp4, A_flat4
                             493.88, 523.25, 466.16, //B4
                             261.63, 277.18, 246.94, //C4
                             293.66, 311.13, 277.18, //D4
                             329.63, 349.23, 311.13, //E4
                             349.23, 369.99, 329.63, //F4
                             392.00, 415.30, 369.99  //G4
                            };
    */

    const double freq[21] = {2272, 2145, 2408,  //A4, A_sharp4, A_flat4
                             2024, 1911, 2145, //B4
                             3822, 3608, 4050, //C4
                             3405, 3214, 3608, //D4
                             3034, 2863, 3214, //E4
                             2863, 2703, 3034, //F4
                             2551, 2408, 2703 //G4
                            };
    while(true) {
        Thread::signal_wait(0x3);
        set_speed(25);
        pwm_duty = 0.7;
        line_mutex.lock();
        sprintf(tx_line, "Speeding up to play tune(5s)\r\n");
        send_line();
        line_mutex.unlock();
        Thread::wait(5000);
        for (int i=0; i<notes_length && run_song; i++) {
            L1L.period_us(freq[notes[i]]*0.5);
            L2L.period_us(freq[notes[i]]*0.5);
            L3L.period_us(freq[notes[i]]*0.5);
            line_mutex.lock();
            sprintf(tx_line, "Playing new note: %d duration %d \r\n", notes[i], durations[i]);
            send_line();
            line_mutex.unlock();
            Thread::wait(1000*durations[i]);
        }
        L1L.period_us(10);
        L2L.period_us(10);
        L3L.period_us(10);
        line_mutex.lock();
        sprintf(tx_line, "Ended Song\r\n");
        send_line();
        line_mutex.unlock();
        sensors.stop();
        output_main = false;
    }
}


//Service available output line by copying to output buffer and triggering Tx
void send_line()
{
    int i;
    char temp_char;
    bool empty;
    i = 0;
    //Start critical section (disabling hardware UART interrupts) 
    // -> avoid RX and TX interrupts from racing to output and input buffers with 
    //    this function 
    NVIC_DisableIRQ(USART2_IRQn);
    empty = (tx_in == tx_out);
    while ((i==0) || (tx_line[i-1] != '\n')) {
        //If output buffer is still not empty, re-enable interrups and wait
        if (((tx_in + 1) % buffer_size) == tx_out) {
            NVIC_EnableIRQ(USART2_IRQn);
            while (((tx_in + 1) % buffer_size) == tx_out) {
            }
            NVIC_DisableIRQ(USART2_IRQn);
        }
        //Copy line to output (circular) buffer
        tx_buffer[tx_in] = tx_line[i];
        i++;
        tx_in = (tx_in + 1) % buffer_size;
    }
    //If PC waiting for a write, must trigger the sending of the first character
    if (pc.writeable() && (empty)) {
        temp_char = tx_buffer[tx_out];
        tx_out = (tx_out + 1) % buffer_size;
        pc.putc(temp_char);
    }
    NVIC_EnableIRQ(USART2_IRQn);
    return;
}


// Read a line from the large rx buffer from rx interrupt routine, behaviour
// similar to send_line
void read_line()
{
    int i;
    i = 0;
    NVIC_DisableIRQ(USART2_IRQn);
    while ((i==0) || (rx_line[i-1] != '\r')) {
        if (rx_in == rx_out) {
            NVIC_EnableIRQ(USART2_IRQn);
            while (rx_in == rx_out) {
            }
            NVIC_DisableIRQ(USART2_IRQn);
        }
        rx_line[i] = rx_buffer[rx_out];
        i++;
        rx_out = (rx_out + 1) % buffer_size;
    }
    NVIC_EnableIRQ(USART2_IRQn);
    rx_line[i-1] = 0;
    return;
}


// INTERRUPT (Triggered by USART): Routine to read in data from serial port
void Rx_interrupt()
{
    char c;
    while ((pc.readable()) && (((rx_in + 1) % buffer_size) != rx_out)) {
        output_main = false;
        c = pc.getc();
        pc.putc(c);
        rx_buffer[rx_in] = c;
        rx_in = (rx_in + 1) % buffer_size;
        if(c == '\r') {
            set_commands.signal_set(0x1);
        }
    }
    return;
}

//INTERRUPT (Triggered by USART):  Routine to write data to the serial port
void Tx_interrupt()
{
    while ((pc.writeable()) && (tx_in != tx_out)) {
        pc.putc(tx_buffer[tx_out]);
        tx_out = (tx_out + 1) % buffer_size;
    }
    return;
}

//THREAD: Control distance (number of rotations) by enabling a PID to controll
//        velocity, once within a target from distance. Once finishd top motor
void distance()
{
    double new_speed;
    int dist_test;

    //Wait for signal from a command
    while(true) {
        Thread::signal_wait(0x2);
        bool test = false;
        bool end_test = false;
        int neg = 1;
        count = 0;
        countold = 0;
        turns = 0;
        if(target_distance<0) {
            neg = -1;
        }
        SetDist.setSetPoint(target_distance);
        //Allow for a new command to cancel the current command
        while(run_dist) {
            dist_test = target_distance - turns;
            if(dist_test*neg < 1 ) {
                end_test = true;
                sensors.stop();
                L1L.write(0);
                L1H = 0;
                L2L.write(0);
                L2H = 0;
                L3L.write(0);
                L3H = 0;
                output_main = false;
                line_mutex.lock();
                sprintf(tx_line, "Ended rotating to %d revolutions\r\n", target_distance);
                send_line();
                line_mutex.unlock();
                run_dist = false;
            }
            if(test == true) {
                //If close to target control speed through PID
                SetDist.setProcessValue(turns);
                new_speed = SetDist.compute();
                a = new_speed;
                set_speed(new_speed);
                Thread::wait(Dist_Rate*1000);
                if(rotation_time == 0) set_speed(50*neg);
            } else {
                //Check distance to target
                if(target_distance*neg > 150 && dist_test*neg < 120) test = true;
                else if(dist_test*neg < target_distance*(2.0/3.0)*neg) test = true;
                else {
                    //Keep rotating constant speed
                    Thread::wait(Dist_Rate*1000);
                }
            }
        }
    }
}