/*
 * main.cpp
 * Laos Controller, main function
 *
 * Copyright (c) 2011 Peter Brier
 *
 *   This file is part of the LaOS project (see: http://wiki.protospace.nl/index.php/LaOS)
 *
 *   LaOS is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   LaOS is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with LaOS.  If not, see <http://www.gnu.org/licenses/>.
 *
 * This program consists of a few parts:
 *
 * ConfigFile   Read configuration files
 * EthConfig    Initialize an ethernet stack, based on a configuration file (includes link status monitoring)
 * LaosDisplay  User interface functions (read keys, display text and menus on LCD)
 * LaosMenu     User interface stuctures (menu navigation)
 * LaosServer   TCP/IP server, accept connections read/write data
 * LaosMotion   Motion control functions (X,Y,Z motion, homing)
 * LaosIO       Input/Output functions
 * LaosFile     File based jobs (read/write/delete)
 *
 * Program functions:
 * 1) Read config file
 * 2) Enable TCP/IP stack (Fixed ip or DHCP)
 * 3) Instantiate tcp/ip port and accept connections
 * 4) Show menus, peform user actions
 * 5) Controll laser
 * 6) Controll motion
 * 7) Set and read IO, check status (e.g. interlocks)
 *
 */
#include "global.h"
#include "ConfigFile.h"
#include "EthConfig.h"
#include "LaosServer.h"
#include "LaosMenu.h"
#include "LaosMotion.h"
#include "SDFileSystem.h"

//
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);


// Status and communication
DigitalOut eth_link(p29); // green
DigitalOut eth_speed(p30); // yellow
EthernetNetIf *eth; // Ethernet, tcp/ip

// Filesystems
LocalFileSystem local("local");   //File System
// SparkFun board  SDFileSystem sd(p5, p6, p7, p8, "sd");
SDFileSystem sd(p11, p12, p13, p14, "sd");

// Laos objects
LaosDisplay *dsp;
LaosMenu *mnu;
LaosServer *srv;
LaosMotion *mot;
int port;

// Externs
extern int x0,y0,z0;

// Protos
void ReadFile(void);


/**
*** Main function
**/
int main() 
{
  printf("BOOT...\n");  
  eth_speed = 1;
  
  printf("MENU...\n"); 
  mnu = new LaosMenu();
  eth_speed=0;
  
  printf("CONFIG...\n");
  eth = EthConfig("/local/config.txt", &port);
  eth_speed=1;
      
  printf("SERVER...\n");
  srv = new LaosServer(port);
  
  printf("MOTION...\n"); 
  mot = new LaosMotion();
  
  printf("RUN...\n");
  
  // Wait for key, and then home
  printf("WAIT FOR HOME KEY...\n");
  
  // Start homing
  while ( !mot->isStart() );
  mot->home();
  // if ( !mot->isHome ) exit(1);
  printf("HOME DONE.\n");
  
  
  while(1) 
  {
    // Get file
    printf("GET FILE...\n");
  
    led1=led2=led3=led4=0;
    ReadFile();
    mot->reset();
    printf("RUN FILE...\n");
    // Read from BIN file
    FILE *in = fopen("/sd/1.bin", "rb");
    int v;
    while (!feof(in))
    { 
      fread(&v, sizeof(v),1,in);   
      while (! mot->ready() );
      mot->write(v);  
    }
    fclose(in);
    // done
    printf("DONE!...\n");
  }
  
  exit(2);
  
  
  // Read from ASCII file
  /* in = fopen("/sd/3.txt", "r");
  while (!feof(in))
  {    
    fscanf(in, "%d", &v);
    while (! mot->ready() );
    mot->write(v);  
  }
  fclose(in);
  */
  
}


/**
*** Read file from network and save on SDcard
**/
void ReadFile(void)
{
  unsigned short int i=0,j=0,done=0,connected=0;
  int c, sign=1;
  Timer tmr;
  tmr.start();
  FILE *out = fopen("/sd/1.bin", "wb");
  char str[16];
  
  while(!done)
  {
    Net::poll();
    if(tmr.read() > 0.01) // sec
    {
      mnu->x = x0;
      mnu->y = y0;
      mnu->Handle();
      if ( mnu->x != x0 || mnu->y != y0 )
      {
        mot->write(0); mot->write(mnu->x); mot->write(mnu->y);
        while(!mot->ready());
      }
      
      tmr.reset();
      if ( i++ & ( EthSpeed() ? 0x400 : 0x800) )
        eth_speed = !eth_speed; //Show that we are alive
      eth_link = EthLink();
    } 
    c =  srv->read();
    if ( c != -2 && !connected) 
    {
      connected=1;
      printf("Connected...");
    }
    switch(c)
    {
      case -2: if ( connected ) done=1; break;
      case '0': case '1': case '2':  case '3':  case '4': 
      case '5': case '6': case '7':  case '8':  case '9':  
        if ( j < 16) 
          str[j++] = (char)c;
        break;
      case '-': sign = -1; break;
      case ' ': case '\t': case '\r': case '\n':
        if ( j )
        {
          int val=0, d=1;
          while(j) 
          {
            if ( str[j-1] == '-' ) 
              d *= -1;
            else
              val += (str[j-1]-'0') * d;
            d *= 10;
            j--;
          }
          val *= sign;
          sign = 1;
          fwrite(&val, sizeof(val), 1, out); 
        }
        break;
    } // Switch
  } // while true
  fclose(out);
} // read file
