/*
 * EthSetup.cpp
 * Setup ethernet interface, based on config file
 * Read IP, Gateway, DNS and port for server.
 * if IP is not defined, or dhcp is set, use dhcp for Ethernet
 *
 * Copyright (c) 2011 Peter Brier
 *
 *   This file is part of the LaOS project (see: http://wiki.protospace.nl/index.php/LaOS)
 *
 *   LaOS is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   LaOS is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with LaOS.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
#include "EthConfig.h"

// Ethernet connection status inputs (from phy) and outputs (to leds)
#define ETH_LINK (1<<25)
#define ETH_SPEED (1<<26)
PortIn eth_conn(Port1, ETH_LINK | ETH_SPEED);   // p25: link, 0=connected, 1=NC, p26: speed, 0=100mbit, 1=10mbit

IPInfo ipInfo;

/**
*** Return Speed status, true==100mbps
**/
bool EthSpeed(void)
{
  int s = eth_conn.read();
  return !(s & ETH_SPEED) && !(s & ETH_LINK);
}

/**
*** Return Link status, true==connected
**/
bool EthLink(void)
{
  int s = eth_conn.read();
  return !(s & ETH_LINK);
}

      

/**
*** Return a IpAddress, from a string in the format: ppp.qqq.rrr.sss
**/
IpAddr IpParse(char *a) 
{
    int n = 0, j, i[4];
    char c[4];

    i[0] = i[1] = i[2] = i[3] =0;
    for (n=0; n<4; n++) {
        while (*a && (*a < '0' || *a > '9'))
            a++;
        j = 0;
        while (*a && *a >= '0' && *a <= '9') 
        {
            if ( j<3 )
                c[j++] = *a;
            a++;
        }
        c[j] = 0;
        i[n] = atoi(c);
    }
    return IpAddr(i[0], i[1], i[2], i[3]);
}


/**
*** EthConfig
**/
EthernetNetIf * EthConfig(char *name, int *port)
 {
    char val[32];
    int dhcp=0;
    EthernetNetIf *eth;
    IpAddr ip, mask, gw, dns;
    printf("\r\nOpen config file: \r\n");
    ConfigFile cfg(name);
    if ( !cfg.IsOpen() ) 
    {
      printf("File does not exists. Using defaults\n");
    }
    cfg.Value("ip", val, sizeof(val), "192.168.12.10");
    printf("ip: '%s'\n\r", val);
    ip = IpParse(val);

    cfg.Value("netmask", val, sizeof(val), "255.255.255.0");
    printf("netmask: '%s'\n\r", val);
    mask = IpParse(val);

    cfg.Value("gateway", val, sizeof(val), "192.168.12.254");
    printf("gateway: '%s'\n\r", val);
    gw = IpParse(val);

    cfg.Value("dns", val, sizeof(val), "192.168.12.254");
    printf("dns: '%s'\n\r", val);
    dns = IpParse(val);

    cfg.Value("port", port, 4000);
    printf("port: '%d'\n\r", *port);
    cfg.Value("dhcp", &dhcp, dhcp);
    printf("dhcp: '%d'\n\r", dhcp);
    if ( dhcp )
        eth = new EthernetNetIf();
    else
        eth = new EthernetNetIf(ip, mask, gw, dns);

    printf("Setup...\n");
    if ( eth->setup() == ETH_TIMEOUT ) 
    {
        printf("Timeout!\n");
    }

    IpAddr myip = eth->getIp();
    ipInfo.ip[0] = myip[0];
    ipInfo.ip[1] = myip[1];
    ipInfo.ip[2] = myip[2];
    ipInfo.ip[3] = myip[3];
    
    printf("mbed IP Address is %d.%d.%d.%d\r\n", myip[0], myip[1], myip[2], myip[3]);
    return eth;
}