/*
 * LaosMenu.cpp
 * Menu structure and user interface. Uses LaosDisplay
 *
 * Copyright (c) 2011 Peter Brier
 *
 *   This file is part of the LaOS project (see: http://wiki.protospace.nl/index.php/LaOS)
 *
 *   LaOS is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   LaOS is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with LaOS.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
#include "LaosMenu.h"

//extern int x0,y0,z0;

static const char *menus[] = {
    "MAIN",        //0 
    "MOVE",        //1
    "FOCUS",       //2
    "HOME",        //3
    "ORIGIN",      //4
    "START JOB",   //5
    "DELETE JOB",  //6 
    "IP",          //7
    "POWER / SPEED",//8
    "IO", //9
};

static const char *screens[] = {
    //0: main, navigate to  MOVE, FOCUS, HOME, ORIGIN, START JOB, IP, DELETE JOB, POWER
#define MAIN (0)
    "$$$$$$$$$$$$$$$$" "\n"
    "<--- 0 ---> [ok]",
  
#define MOVE (MAIN+1)
    "X: +6543210 MOVE" "\n"
    "Y: +6543210 [ok]",
    
#define FOCUS (MOVE+1)
    "Z: +543210 FOCUS" "\n"
    "            [ok]",

#define HOME (FOCUS+1)
    "     HOME?      " "\n"
    "[cancel]    [ok]",  

#define ORIGIN (HOME+1)
    "  SET ORIGIN?   " "\n"
    "[cancel]    [ok]",   

#define RUN (ORIGIN+1)
    "RUN: $$$$$$$$$$$" "\n"
    "[cancel]<10><ok>",   

#define DELETE (RUN+1)
    "DEL: $$$$$$$$$$$" "\n"
    "[cancel]<10>[ok]",   

#define IP (DELETE+1)
    "210.210.210.210 " "\n"
    "$$$$$$$$<0> [ok]",    

#define POWER (IP+1)
    "$$$$$$$: 6543210" "\n"
    "[cancel] <0>[ok]",    

#define IO (POWER+1)
    "$$$$$$$$$$$=0 IO" "\n"
    "[cancel]    [ok]",    

// Intermediate screens
#define DELETE_OK (POWER+1)
    "   DELETE 10?   " "\n"
    "<cancel>    [ok]",   

#define HOMING (DELETE_OK+1) 
    "HOMING...6543210" "\n"
    "[cancel]        ",   
    
#define BUSY (HOMING+1)
    "BUSY: $$$$$$$$$$" "\n"
    "[cancel]    [ok]",

#define PAUSE (BUSY+1)
    "PAUSE: $$$$$$$$$" "\n"
    "[cancel]    [ok]",

};

static  const char *ipfields[] = { "IP", "NETMASK", "GATEWAY", "DNS" };
static  const char *powerfields[] = { "Pmin %", "Pmax %", "Voff", "Von" };
static  const char *iofields[] = { "o1:PURGE", "o2:EXHAUST", "o3:PUMP", "i1:COVER", "i2:PUMPOK", "i3:LASEROK", "i4:PURGEOK" };
  
  
/**
*** Make new menu object
**/
// LaosMenu::LaosMenu(const LaosDisplay &display) : dsp(display) 
LaosMenu::LaosMenu() 
{
  waitup=timeout=0;
  sarg = NULL;
  x=y=z=0;
  screen=prevscreen=menu=speed=0;
  dsp.cls();
}



/**
*** Destroy menu object
**/
LaosMenu::~LaosMenu() 
{

}



/**
*** Handle menu system
*** Read keys, an plan next action on the screen, output screen if something changed
**/
void LaosMenu::Handle()
{
  static int count=0;
  int c = dsp.read();
  if ( count++ > 10) count = 0;
  if ( c )
    timeout = 10;
  else
    if ( timeout ) timeout--;
  if ( screen != prevscreen ) 
    waitup = 1;
  if ( waitup && timeout) // if we have to wait for key-up, cancel the keypress
    c = 0;
  if ( waitup && !timeout )
    waitup=0;
  if ( !timeout )  // increase speed if we keep button pressed longer 
    speed = 1;
  else
  {
    speed += 1;
    if ( speed >= 500 ) speed = 100;    
  }

  if ( c || screen != prevscreen || count >9 ) 
  {
    
    switch(c) // screen independent handling
    { 
      case K_FUP: screen=FOCUS; break;
      case K_FDOWN: screen=FOCUS; break;
      case K_ORIGIN: screen=ORIGIN; break;
    }
    
    prevscreen = screen;  
    
    switch ( screen ) 
    {
      case MAIN:
        switch ( c )
        {
          case K_RIGHT: menu+=1; waitup=1; break;
          case K_LEFT: menu-=1; waitup=1; break;
          case K_UP: screen=MOVE; break;
          case K_DOWN: screen=MOVE; break;
          case K_OK: screen=menu; break;
          case K_CANCEL: menu=MAIN; break;
        }
        menu =  menu % (sizeof(menus) / sizeof(menus[0]));
        sarg = (char*)menus[menu]; 
        args[0] = menu;
        break;
      
      case MOVE: // pos xy
        switch ( c )
        {
          case K_UP: y+=speed; break;
          case K_DOWN: y-=speed; break;
          case K_LEFT: x-=speed; break;
          case K_RIGHT: x+=speed; break;
          case K_OK: case K_CANCEL: screen=MAIN; waitup=1; break;
        }
        args[0]=x; args[1]=y; 
        break;
        
      case FOCUS: // focus
        switch ( c )
        {
          case K_FUP: case K_UP: z+=speed; break;
          case K_FDOWN: case K_DOWN: z-=speed; break;
          case K_LEFT: screen=MAIN; menu -= 1; break;
          case K_RIGHT: screen=MAIN; menu += 1; break;
          case 0: break;
          default: screen=MOVE; break;
        }
        args[0]=z;  
        break;

      case HOME:// home
        switch ( c )
        {
          case K_OK: x=0; y=0; screen=HOMING; break;
          case K_CANCEL: screen=MOVE; waitup=1; break;
        }
        break;
  
      case ORIGIN: // origin
        switch ( c )
        {
          case K_OK: x=0; y=0; screen=MOVE; break;
          case K_CANCEL: screen=MOVE;  break;
        }
        break;
        
      case IP: // IP
        switch ( c )
        {
          case K_RIGHT: ipfield++; waitup=1; break;
          case K_LEFT: ipfield--; waitup=1; break;
          case K_OK: screen=MAIN; break; 
          case K_CANCEL: screen=MAIN; break;
        }
        ipfield %= 4;
        sarg = (char*)ipfields[ipfield];
        args[4] = ipfield;
        switch(ipfield)
        {
          case 0: memcpy(args, ipInfo.ip, 4*sizeof(int) ); break;
          default: memset(args,0,4*sizeof(int)); break;
        }
        break; 
        
       case IO: // IO
        switch ( c )
        {
          case K_RIGHT: iofield++; waitup=1; break;
          case K_LEFT: iofield--; waitup=1; break;
          case K_OK: screen=MAIN; break; 
          case K_CANCEL: screen=MAIN; break;
        }
        iofield %= sizeof(iofields)/sizeof(char*);
        sarg = (char*)iofields[iofield];
        args[0] = ipfield;
        args[1] = ipfield;
        break;       
        
      case POWER: // POWER
        switch ( c )
        {
          case K_RIGHT: powerfield++; waitup=1; break;
          case K_LEFT: powerfield--; waitup=1; break;
          case K_UP: power[powerfield % 4] += speed;  break;
          case K_DOWN: power[powerfield % 4] -= speed;  break;
          case K_OK: screen=MAIN;  break; 
          case K_CANCEL: screen=MAIN;  break;
        }
        powerfield %= 4;
        args[1] = powerfield;
        sarg = (char*)powerfields[powerfield];
        args[0] = power[powerfield];
        break;     


      case HOMING: // Homing screen
        switch ( c )
        {
          case K_CANCEL: screen=MAIN; break;
        }
        args[0]++;
        break;     
        
      default: screen = MAIN; break;    
    }  
    dsp.ShowScreen(screens[screen], args, sarg);
  }
  
}
