/**
 * LaosDisplay.cpp
 * User interface class, for 2x16 display and keypad
 *
 * Copyright (c) 2011 Peter Brier
 *
 *   This file is part of the LaOS project (see: http://wiki.protospace.nl/index.php/LaOS)
 *
 *   LaOS is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   LaOS is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with LaOS.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Assume (USB) serial connected display (or terminal) in future, this could be CAN or I2C connected
 *
 */
#include "LaosDisplay.h"


// I2C
I2C i2c(p9, p10);        // sda, scl
#define _I2C_ADDRESS 0x04
#define _I2C_HOME 0xFE
#define _I2C_CLS 0xFF
#define _I2C_BAUD 10000

  
// Make new config file object
LaosDisplay::LaosDisplay()
{
  serial = new Serial(USBTX, USBRX);
  serial->baud(115200);
}


//Write char
 void LaosDisplay::write(char c) 
{
  serial->putc(c);
}

// Write string
void LaosDisplay::write(char *s) 
{
  if ( s == NULL ) return;
  i2c.write(_I2C_ADDRESS, s, strlen(s));
  while (*s)
    serial->putc(*s++);
}

// Clear screen
void LaosDisplay::cls() 
{
    write('\t');
}


// Read Key
int LaosDisplay::read() 
{
  char key;
  int result;
  result = i2c.read(_I2C_ADDRESS,&key, 1);
  if ( result ) return key;
  if ( serial->readable() )
    return serial->getc();
  else
    return 0;
}



/**
*** Screens are defined with:
*** name, line[2], int* i[4], char *s;
***
*** in the lines, the following replacements are made:
*** $$$$$$ : "$" chars are replaced with the string argument for this screen (left alligned), if argument is NULL, spaces are inserted
*** +9876543210: "+" and numbers are replaced with decimal digits from the integer arguments (the "+" sign is replaced with '-' if the argument is negative
*** 
***
**/
inline char digit(int i, int n)
{
  int d=1;
  char c;
  if ( i<0 ) i *= -1;
  while(n--) d *= 10;
  c = '0' + ((i/d) % 10);
  return c;
}
void LaosDisplay::ShowScreen(const char *l, int *arg, char *s)
{
  char c, next=0,surpress=1;
  cls(); 
  while ( (c=*l) )
  {
    switch ( c )
    {
    case '$': 
      if (s != NULL && *s)
        c = *s++;
      else 
        c = ' ';
      break;
      
    case '+':
      if (arg != NULL && *arg < 0)
        c = '-';
      else 
        c = '+';
      break;
        
    case '0': next=1; surpress=0; 
    case '1': case '2': case '3': case '4': 
    case '5': case '6': case '7': case '8': case '9':
      char d = ' ';
      if (arg != NULL )
      {
        d = digit(*arg, *l-'0');
        if ( d == '0' && surpress  ) // supress leading zeros 
          d =  ' ';
        else
          surpress = 0;
        if ( next && arg != NULL ) // take next numeric argument
        {
          arg++;
          next=0;
          surpress=1;
        }
      }
      c = d;
      break;
    }
    write(c);
    l++;
  }
}


// EOF