/*
 * main.cpp
 *
 * Copyright (c) 2011 Peter Brier
 *
 *   This file is part of the LaOS project (see: http://wiki.protospace.nl/index.php/LaOS)
 *
 *   LaOS is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   LaOS is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with LaOS.  If not, see <http://www.gnu.org/licenses/>.
 *
 * This program implements Bresenham line algorithm program (2D)
 * see: http://en.wikipedia.org/wiki/Bresenham%27s_line_algorithm
 * In addition: we controll a Z-axis and a laser.
 * Note: Fixed speed lasering (no accelleration).
 * Simple program, no line buffering. Considerable jitter on pulses. Speed varies approx 30%, depending on angle (varies over 1.5 degree)
 *
 */
#include "mbed.h"

// serial comm
Serial pc(USBTX, USBRX); 

// status leds
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);

// motors
DigitalOut xdir(p26);
DigitalOut xstep(p25);
DigitalOut ydir(p24);
DigitalOut ystep(p23);
DigitalOut zdir(p22);
DigitalOut zstep(p21);

// laser
DigitalOut laser(p27);

// globals
Ticker ticker;  // Timer

// Besenham variables
int state, laseron;
int x0,y0,x1,y1,z0,z1;  // positions: 0=current, 1=new
int dx, sx; // step
int dy, sy;  // delta
int err, e2; // error


/**
*** ticker: implement timer ticker, step if required
**/
void tick() 
{
  // Control X,Y: line tracker
  switch ( state )
  {
    case 0: // idle (make sure laser is off)
      laser = laseron =  0;
      break;
    case 1: // init
      dx = abs(x1-x0);
      sx = x0<x1 ? 1 : -1;
      dy = abs(y1-y0);
      sy = y0<y1 ? 1 : -1; 
      err = (dx>dy ? dx : -dy)/2;
      xdir = sx > 0;
      ydir = sy > 0;
      state++;
      laser = laseron;
      break;
    case 2: // moving
      if (x0==x1 && y0==y1) 
        state = 0;
      else
      {
        e2 = err;
        if (e2 >-dx) { err -= dy; x0 += sx; xstep = 1;}
        if (e2 < dy) { err += dx; y0 += sy; ystep = 1;}
      }
      break;
  }
  
  // Controll Z: tracking controller (independent of XY motion, not queued)
  if ( z1 != z0 ) 
  {
    zdir = z1 > z0;
    wait_us(5);
    zstep= 1;
    if ( z1 > z0 ) z0++;
    else z0--;
  }
  
  if ( xstep == 1 || ystep == 1 || zstep == 1 )
  {
    wait_us(5);
    xstep = ystep = zstep = 0;
  }
  
  // diagnostic info
  led1 = xdir;
  led2 = ydir;
  led3 = zdir;
  led4 = laser;
} // ticker



/**
*** main()
*** Read commands, report state
**/
int main() 
{
  int v=100,c=0,z=0;

  pc.baud(115200);  
  ticker.attach_us(&tick, v); // the address of the function to be attached (flip) and the interval (microseconds)

  // init
  state = laser = xstep = xdir = ystep = ydir = zstep = zdir = 0;
  
  while( 1 )
  {    
     scanf("%d", &c); 
     switch( c )
     {
          case 0: // move x,y (laser off)
            scanf("%d %d", &x1, &y1);
            laseron = 0;
            state = 1; 
            break;
          case 1: // line x,y (laser on)
            scanf("%d %d", &x1, &y1);
            laseron = 1;
            state = 1; 
            break;
          case 2: // move z 
            scanf("%d", &z1); 
            break;
          case 3: // set t (1/v)
            scanf("%d", &v);
            ticker.attach_us(&tick, v);
            break;
         case 4: // set x,y,z absolute
            scanf("%d %d %d", &x0, &y0, &z);
            z1 = z0 = z;
         case 5: // nop
           break;
         default: // I do not understand: stop motion
            state = 0;
          break;
    }
    printf("%d %d %d %d %d %d\r\n", state, x0, y0, z0, v, laseron);
  }
}
