/* Test harness for TextStar serial LCD library
 * Paul Griffith
 * Last edit: 10 Jan 2010
 */
 
#include "mbed.h"
#include "TextStar.h"

void bargraph_demo(int);
void printhelps(void);
void bad_args(void);

Serial pc(USBTX, USBRX);
TextStar lcd(p9, p10); // tx, rx

int main() {
    char buf[120], *cp;
    char arg1[50], arg2[50], arg3[50];
    int argc, cnt = 0, i, j, k, l;


    printf("TextStar Serial LCD test harness\n");
    printf("Type ? for help\n");
    lcd.printf("TextStar Serial\rLCD test harness");
            
    while (1) {
        printf("> ");
        cp = buf;
        while ( (*cp++ = putchar(getchar())) != '\r') ;   //get a line of input
        *cp = '\0';                                       //terminate buffer
        printf("\n");
        argc = sscanf(buf, "%s%s%s", arg1, arg2, arg3);   //extract command and arguments
        if (argc < 1)
            continue;
        switch (arg1[0]) {

            case 'b':        //bar graph demo
                bargraph_demo(0);
                wait(1.0);
                bargraph_demo(1);
                break;

            case 'c':        //clear display
                lcd.cls();
                break;

            case 'C':        //set cursor style
                if (argc != 2) {
                    bad_args();
                    break;
                }
                sscanf(arg2, "%d", &i);   //extract style number
                lcd.set_cursor(i);                
                break;

            case 'd':    //move cursor down
                j = 1;
                if (argc == 2)
                    sscanf(arg2, "%d", &j);
                for (i = 0; i < j; i++)
                    lcd.down();
                break;
                                
            case 'D':    //define custom character
                sscanf(buf, "%s%d%x%x", arg1, &i, &j, &k);   //extract arguments
                arg3[0] = (j >> 24) & 0xFF;
                arg3[1] = (j >> 16) & 0xFF;
                arg3[2] = (j >> 8) & 0xFF;
                arg3[3] = j & 0xFF;
                arg3[4] = (k >> 24) & 0xFF;
                arg3[5] = (k >> 16) & 0xFF;
                arg3[6] = (k >> 8) & 0xFF;
                arg3[7] = k & 0xFF;
                lcd.custom_char(i, arg3);
                lcd.cls();
                lcd.printf("Custom chars");
                lcd.locate(0, 1);
                lcd.printf("128-135 %c%c%c%c%c%c%c%c", 0x80, 0x81,
                    0x82, 0x83, 0x84, 0x85, 0x86, 0x87);
                break;

            case 'e':        //enter text at current position
                if (argc < 2) {
                    bad_args();
                    break;
                }
                *--cp = '\0';    //get rid of \r at end of buffer
                sscanf(buf, "%s%n", arg1, &i);   //extract arguments
//                cp = buf + i;
//                while (*cp++ == ' ') ;    //find start of text
//                --cp;
                cp = buf + i + 1;    //allow leading white space
                lcd.printf("%s", cp);
                break;

            case 'h':        //cursor home
                lcd.home();
                break;

            case 'k':        //get key states
                while (lcd.readable() )
                    lcd.getc();        //empty buffer
                printf("Key states = ");
                lcd.send_keys();
                i = lcd.getc();
                j = lcd.getc();
                k = lcd.getc();
                l = lcd.getc();
                printf("%c%c%c%c\n", i, j, k, l);
                break;

            case 'K':        //get key output until any char typed on pc
                printf("Key output: ");
                while (pc.readable() == 0) {
                    if (lcd.readable() )
                        pc.putc(lcd.getc() );
                }
                while (pc.readable() )
                    pc.getc();
                printf("\nAll done\n");
                break;
                
            case 'o':        //output demo
                lcd.cls();
                j = 20;
                if (argc == 2) {
                    sscanf(arg2, "%d", &j);
                }
                for (i = 0; i < j; i++) {
                    lcd.printf("Count = %d\r", cnt++);
                    wait( (i == 0) ? 1.0 : 0.2);
                }
                break;

            case 'l':        //move cursor left
                j = 1;
                if (argc == 2)
                    sscanf(arg2, "%d", &j);
                for (i = 0; i < j; i++)
                    lcd.left();
                break;
                            
            case 'p':        //set cursor position
                if (argc < 3) {
                    bad_args();
                    break;
                }
                sscanf(buf, "%s%d%d", arg1, &i, &j);   //extract arguments
                lcd.locate(j, i);
                break;
                
            case 'q':        //quit
                exit(1);
                break;

            case 'r':        //move cursor right
                j = 1;
                if (argc == 2)
                    sscanf(arg2, "%d", &j);
                for (i = 0; i < j; i++)
                    lcd.right();
                break;
                
            case 'R':        //reset display (and set autoblank (and autoscroll) )
                if (argc == 1) {
                    lcd.reset();
                }
                else if (argc == 2) {
                    sscanf(arg2, "%d", &i);
                    lcd.reset(i);
                }
                else if (argc >= 3) {
                    sscanf(arg2, "%d", &i);
                    sscanf(arg3, "%d", &j);
                    lcd.reset(i, j);
                }
                break;

            case 's':        //scroll down
                j = 1;
                if (argc == 2)
                    sscanf(arg2, "%d", &j);
                for (i = 0; i < j; i++)
                    lcd.scroll_down();
                break;

            case 'S':        //scroll up
                j = 1;
                if (argc == 2)
                    sscanf(arg2, "%d", &j);
                for (i = 0; i < j; i++)
                    lcd.scroll_up();
                break;
                
            case 'u':        //move cursor up
                j = 1;
                if (argc == 2)
                    sscanf(arg2, "%d", &j);
                for (i = 0; i < j; i++)
                    lcd.up();
                break;
                   
            case 'v':        //display firmware version
                lcd.version();
                break;

                            
            case 'w':        //move visible window
                if (argc != 2) {
                    bad_args();
                    break;
                }
                sscanf(arg2, "%d", &i);   //extract line number
                lcd.window(i);
                break;
                
            case 'x':        //send hex data to display
                argc = sscanf(buf, "%s%x%x%x%x", arg1, &i, &j, &k, &l);   //extract arguments
                if (argc > 1)
                    lcd.putc(i & 0xFF);
                if (argc > 2)
                    lcd.putc(j & 0xFF);
                if (argc > 3)
                    lcd.putc(k & 0xFF);
                if (argc > 4)
                    lcd.putc(l & 0xFF);                    
                break;

            case '?':        //print help
                printhelps();
                break;
                
            default:
                printf("?? Unknown command\n");
                break;
        }
    }
}

//Command functions

void printhelps(void) {
    printf("Command summary:\n");
    printf("b            bar graph demo\n");
    printf("c            clear display\n");
    printf("C style      set cursor style (0-4)\n");
    printf("d [n]        cursor down n times (default 1)\n");
    printf("D n bm1 bm2  define custom char n (128-135) with bitmap in hex\n");
    printf("             bm1 = bitmap rows 1-4, bm2 = rows 5-8\n");
    printf("e text       enter text at current position\n");
    printf("h            cursor home\n");
    printf("k            get key states\n");
    printf("K            show key output\n");
    printf("l [n]        cursor left n times (default 1)\n");
    printf("o            output demo\n");
    printf("p line col   set cursor position to line (0-15) and column (0-15)\n");
    printf("q            quit\n");
    printf("r [n]        cursor right n times (default 1)\n");
    printf("R [ab] [as]  reset display, set auto-blank (0/1) and auto-scroll (0/1)\n");
    printf("s [n]        scroll down n times (default 1)\n");
    printf("S [n]        scroll up n times (default 1)\n");
    printf("u [n]        cursor up n times (default 1)\n");
    printf("v            display firmware version\n");
    printf("w line       move display window to specified line (0-14)\n");
    printf("x b b b b    send hex data to display, up to 4 bytes\n");
}

void bargraph_demo(int capped) {

    int i;

    lcd.cls();
    lcd.printf("BAR GRAPH DEMO %d", capped + 1);
    for (i = 0; i <= 100; i++) {
        lcd.locate(0, 1);
        lcd.bar_graph(capped, 16, i);
        wait( (i == 0) ? 1.0 : 0.02);
    }
    wait(1.0);
    for (i = 100; i >= 0; i--) {
        lcd.locate(0, 1);
        lcd.bar_graph(capped, 16, i);
        wait( (i == 0) ? 1.0 : 0.02);
    }
}

void bad_args(void) {
    printf("?? Bad arguments\n");
}

