/******************************************************************************
 * File:      FastOut.h
 * Author:    Paul Griffith
 * Created:   18 Jun 2011
 * Last Edit: 29 Sep 2013
 * Version:   1.10
 *
 * Description:
 * High-speed digital output class for mbed
 *
 * Usage: FastOut<PinName> name;
 * PinName uses the same assignment operators as with DigitalOut
 * Example: FastOut<LED2> led2;
 * A simple loop as below takes 4 clocks. This gives a 24MHz output waveform
 *      while (1) { out = 1; out = 0; } 
 *
 * Class code by Igor Skochinsky. Copied from posting by Sylvain Azerian:
 *     http://sylvain.azarian.org/doku.php?id=mbed.
 *
 * Copyright (c) 2011-2013 Paul Griffith, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 *
 * Modifications:
 * Ver  Date    By  Details
 * 1.00 18Jun11 PG  File created.
 * 1.01 20Jun11 PG  Tidied up, added header block.
 * 1.10 29Sep13 PG  Replaced link to MIT Licence with full MIT Licence text.
 *
 ******************************************************************************/

#ifndef MBED_FASTOUT_H
#define MBED_FASTOUT_H

#include "mbed.h"

template <PinName pin> class FastOut {
// pin = LPC_GPIO0_BASE + port * 32 + bit
// port = (pin - LPC_GPIO0_BASE) / 32
// bit  = (pin - LPC_GPIO0_BASE) % 32

    // helper function to calculate the GPIO port definition for the pin
    // we rely on the fact that port structs are 0x20 bytes apart
    inline LPC_GPIO_TypeDef* portdef() {
        return (LPC_GPIO_TypeDef*)(LPC_GPIO0_BASE + ((pin - P0_0)/32)*0x20);
    };

    // helper function to calculate the mask for the pin's bit in the port
    inline uint32_t mask() {
        return 1UL<<((pin - LPC_GPIO0_BASE) % 32);
    };

public:
    FastOut() {
        // set FIODIR bit to 1 (output)
        portdef()->FIODIR |= mask();
    }
    void write(int value) {
        if ( value )
            portdef()->FIOSET = mask();
        else
            portdef()->FIOCLR = mask();
    }
    int read() {
        return (portdef()->FIOPIN) & mask() != 0;
    }
    FastOut& operator= (int value) {
        write(value);
        return *this;
    };
    FastOut& operator= (FastOut& rhs) {
        return write(rhs.read());
    };
    operator int() {
        return read();
    };
};

#endif

// END of FastOut.h
