#include "mbed.h"
#include "7SegSRDriver.h"

/* mbed Simple 'Roulette' Game
 * Making use of 7-Segment Display Driver Library (via an 8bit Shift Register)
 * Copyright (c) 2011 Paul Law
 *
 * User chooses a number between 1 and 9 by pressing button A repeatedly
 * - Selection is shown on second 7seg display
 * User then presses button B to begin the game
 * - Number between 1 and 9 is chosen at random
 * - First display shows 'suspense' animation then shows number
 * - If the numbers match the 2 displays flash and the blue leds flash
 * - If the numbers don't match the 1st display flashes only
 * - Another round of the game starts, ad infinitum
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

// Pin outs
// p17 - Play button +ve
// p18 - Select button +ve
// p22 - 1st Display Shift Register - Clock
// p23 - 1st Display Shift Register - Latch
// p24 - 1st Display Shift Register - Data
// p25 - 2nd Display Shift Register - Clock
// p26 - 2nd Display Shift Register - Latch
// p27 - 2nd Display Shift Register - Data
DigitalIn playButton(p17);
DigitalIn selectButton(p18);
SSegSRDriver lDisplay(p23,p21,p22, SSegSRDriver_COMN_ANODE);
SSegSRDriver rDisplay(p27,p25,p26, SSegSRDriver_COMN_ANODE);
DigitalOut leds[] = { (LED1),(LED2),(LED3),(LED4) };

// Display Roll Definitions (first element in array is number of items, followed by items
#define ROLL_CWURC  0
#define ROLL_CCWLC  1
#define ROLL_CCWLRC 2
#define ROLL_CWUC   3
#define ROLL_FIG8   4
#define ROLL_CWO    5
unsigned char rollDefs[6][9] = {
    // A clockwise upper reverse C - abg
    {3,2,4,128},
    // An anticlockwise lower C - ged
    {3,128,32,16},
    // An anticlockwise lower reverse C - dcg
    {3,16,8,128},
    // A clockwise upper C - gfa
    {3,128,64,2},
    // A figure of 8 roll - abgedcgf
    {8,2,4,128,32,16,8,128,64},
    // A clockwise O roll - abcdef
    {6,2,4,8,16,32,64}
};

// Performs a single roll animation on the given display
void rollDisplay(SSegSRDriver *display, unsigned char rollType, unsigned char repeats) {
    for (int r=0; r<repeats; r++) {
        for (int i=1; i<=rollDefs[rollType][0]; i++) {
            display->write_raw(rollDefs[rollType][i]);
            wait_ms(75);
        }
    }
    display->clear();
}

int main() {

    unsigned char guess;

    while (1) {

        lDisplay.clear();
        rDisplay.clear();

        // New game animation (figure of 8 across two displays)
        rollDisplay(&rDisplay,ROLL_CWURC,1);
        rollDisplay(&lDisplay,ROLL_CCWLC,1);
        rollDisplay(&rDisplay,ROLL_CCWLRC,1);
        rollDisplay(&lDisplay,ROLL_CWUC,1);

        // User selects guess
        guess = 1;
        rDisplay.write(guess);
        while (!playButton) {
            if (selectButton) {
                guess++;
                if (guess==10) guess=1;
                rDisplay.write(guess);
                while (selectButton) {
                    wait(0.25);
                }
            }
            wait(0.25);
        }

        // Choose random number and animate 'suspense'
        int target = (rand() % 9)+1;
        rollDisplay(&lDisplay,ROLL_CWO,2);

        // Show number and flash (both and leds if win, just target if lost)
        for (int i=0; i<5; i++) {
            lDisplay.clear();
            if (target == guess) {
                rDisplay.clear();
                leds[0] = leds[1] = leds[2] = leds[3] = 1;
            }
            wait_ms(250);
            lDisplay.write(target);
            if (target == guess) {
                rDisplay.write(guess);
                leds[0] = leds[1] = leds[2] = leds[3] = 0;
            }
            wait_ms(250);
        }

        // End of this game round
    }
}
