/* Gestion carte GPS au format XBEE
 * Fournisseur : Embedded Artists - Réf : EA-ACC-023
 *      https://www.embeddedartists.com/products/acc/acc_gps.php
 * Utilise Chipset MT3339 (Mediatek Labs)
 * Décode et affiche trame $GPRMC ou $GPGGA - affiche les autres trames non décodées
 * Caractères reçus sous interruption (UART1-RX)
 *
 * V2 14/3/2018 - ajout décodage trame GGA
 * V3 7/8/2018  - DEBUG AFFICHAGE RMC + GGA / suppression librairies inutiles
 
 * Exemple d'utilisation class GpsXbee (IRQ compatible) :
@code
#include "mbed.h"
#include <stdio.h>
#include "gps-ea-acc023.h"

using namespace gpsxbee_MT3339;
static GpsXbee gpsxbee;
Serial pc2(USBTX, USBRX);           // Port USB vers PC pour debug
DigitalOut led4_GPSON(LED4);    //clignote à chaque trame GPS (1 = allume)

// affichage données GPS quand trame reçue complète au format NMEARMC ou NMEAGGA
main()
{
    pc2.baud(115200); // liaison vers PC DEBUG
    gpsxbee.GpsXbeeInit();
    while(1) {
        if (gpsxbee.gpsTrameRMC == gpsxbee.TRAME_OK && gpsxbee.gpsTrameGGA == gpsxbee.TRAME_OK) {
            gpsxbee.gpsTrameRMC = gpsxbee.TRAME_WAIT;
            printf(" RMC = %s\r\n",gpsxbee.gpsDataRMC);

            gpsxbee.gpsTrameGGA = gpsxbee.TRAME_WAIT;
            printf(" GGA = %s\r\n",gpsxbee.gpsDataGGA);
            led4_GPSON = !led4_GPSON;
        }
    }
}
@endcode

 *  Résultat  = Affichage sur terminal PC (115200 bauds) : Trames décodées :
        RMC : 08/06/18:10H34mn32s - Lat. = 44.957730N - Long. = -0.666333W
        GGA : 08H34mn32s --- Latitude = 44.957730N  --- Longitude = -0.666333W - Nb sat = 9

 */

#ifndef GPSXBEE_H_
#define GPSXBEE_H_

namespace gpsxbee_MT3339 {

class GpsXbee {

public:
    void GpsXbeeInit(void);
    virtual ~GpsXbee();
    void InitRMC(void);
    void InitGGA(void);

    void parseRMC(char* data, int dataLen); // analyse trame GPS
    int  parseData(char* data, int len);
    void lecture(); // Remplissage gpsBuf
    void parseGGA(char* data, int dataLen);


    // Conversion format ddmm.mmmm en mode degrés
    double  getLatitudeAsDegrees(double l,char ns);
    double  getLongitudeAsDegrees(double l,char ns);

    enum NmeaSentenceValue { // type de protocole de message reçu
        NMEA_INVALID    = 0,
        NmeaGga         = 0x01,
        NmeaGsa         = 0x02,
        NmeaGsv         = 0x04,
        NmeaRmc         = 0x08,
        NmeaVtg         = 0x10,
    };
    NmeaSentenceValue NmeaSentence;

    enum gpsTrameValue {
        TRAME_WAIT = 0,     // Trame lue, attente nouvelle trame
        TRAME_IN   = 1,     // Reception Trame en cours
        TRAME_OK   = 2,     // Trame complète prête à lire
        TRAME_ERR  = -1,    // Erreur Trame
    };
    gpsTrameValue  gpsTrameRMC,gpsTrameGGA;
    
    // Message reçu au format $GPRMC
    int rmc_hours;          /** UTC time - hours */
    int rmc_minutes;        /** UTC time - minutes */
    int rmc_seconds;        /** UTC time - seconds */
    int rmc_milliseconds;   /** UTC time - milliseconds */
    char rmc_warning;       /** A = données valides - V = invalides */
    double rmc_latitude;    /** The latitude in ddmm.mmmm format (d = degrees, m = minutes) */
    double rmc_longitude;   /** The longitude in dddmm.mmmm format */
    char rmc_nsIndicator;   /** North / South indicator */
    char rmc_ewIndicator;   /** East / West indicator */
    double rmc_speedKnots;  /** speed in Knots */
    int rmc_jours ;         /** date jjmmaa */
    int rmc_mois ;
    int rmc_ans ;

    // Message reçu au format $GPGGA
    int gga_hours;
    int gga_minutes;
    int gga_seconds;
    int gga_milliseconds;
    double gga_latitude;
    double gga_longitude;
    char gga_nsIndicator;
    char gga_ewIndicator;
    int gga_fix;            //  0 = Fix not available     1 = GPS fix   2 = Differential GPS fix
    int gga_satellites;     /** Number of used satellites */
    double gga_hdop;        /** Horizontal Dilution of Precision */
    double gga_altitude;    /** antenna altitude above/below mean sea-level */
    double gga_geoidal;     /** geoidal separation */

    char gpsDataRMC[200];      /** buffer pour affichage data RMC*/
    char gpsDataGGA[200];      /** buffer pour affichage data GGA*/

private :
    #define MTK3339_BUF_SZ 200      // Taille gpsBuf
    char gpsBuf[MTK3339_BUF_SZ];    // buffer de réception trame GPS

    enum gpsStateValue {
        StateStartGps = 0x30,
        StateDataGps = 0x31,
    };
    gpsStateValue gpsState;         // état réception

    int  gpsBufPos;                 // position index lecture buffer

}; // end class
}  // end namespace
#endif /* GPSXBEE_H_ */
