/*
 * FreeModbus Libary: BARE Demo Application
 * Copyright (C) 2006 Christian Walter <wolti@sil.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * File: $Id: demo.c,v 1.1 2006/08/22 21:35:13 wolti Exp $

 * modified from: Thanassis Mavrogeorgiadis 5-12-2011
 * modified by Yuji Hosogaya 3-16-2012
 */
#include "mbtcp.h"

Serial pc(USBTX, USBRX);

/* ----------------------- Defines ------------------------------------------*/
#define REG_INPUT_START 1001
#define REG_INPUT_NREGS 4

#define REG_HOLDING_START       2001
#define REG_HOLDING_NREGS       130

#define REG_COIL_START (3000+1)
#define REG_COIL_NREGS 8
#define REG_COIL_BYTES REG_COIL_NREGS/8

#define REG_DISC_START (4000+1)
#define REG_DISC_NREGS 8
#define REG_DISC_BYTES REG_DISC_NREGS/8

#define SLAVE_ID 0x0A

/* ----------------------- Static variables ---------------------------------*/
static USHORT   usRegInputStart = REG_INPUT_START;
static USHORT   usRegInputBuf[REG_INPUT_NREGS]={0x0123,0x4567,0x89AB,0xCDEF};
static USHORT   usRegHoldingStart = REG_HOLDING_START;
static USHORT   usRegHoldingBuf[REG_HOLDING_NREGS]={0x0123,0x4567,0x89AB,0xCDEF,0xDEAD,0xBEEF,0xDEAD,0xBEEF,0xDEAD,0xBEEF};
static USHORT    usRegCoilStart = REG_COIL_START;
static UCHAR     usRegCoilBuf[REG_COIL_BYTES]={0xA5};
static USHORT    usRegDiscStart = REG_DISC_START;
static UCHAR     usRegDiscBuf[REG_DISC_BYTES]={0x5A};

/* ----------------------- Start implementation -----------------------------*/

int main() {
    pc.baud(115200);
    mb_init();

    while(1) {
        device_poll();
    }
}

eMBErrorCode
eMBRegInputCB( UCHAR * pucRegBuffer, USHORT usAddress, USHORT usNRegs )
{
    eMBErrorCode    eStatus = MB_ENOERR;
    int             iRegIndex;

    if( ( usAddress >= REG_INPUT_START )
        && ( usAddress + usNRegs <= REG_INPUT_START + REG_INPUT_NREGS ) )
    {
        iRegIndex = ( int )( usAddress - usRegInputStart );
        while( usNRegs > 0 )
        {
            *pucRegBuffer++ = ( unsigned char )( usRegInputBuf[iRegIndex] >> 8 );
            *pucRegBuffer++ = ( unsigned char )( usRegInputBuf[iRegIndex] & 0xFF );
            iRegIndex++;
            usNRegs--;
        }
    }
    else
    {
        eStatus = MB_ENOREG;
    }
    return eStatus;
}

eMBErrorCode
eMBRegHoldingCB( UCHAR * pucRegBuffer, USHORT usAddress, USHORT usNRegs, eMBRegisterMode eMode )
{
    eMBErrorCode    eStatus = MB_ENOERR;
    int             iRegIndex;

    if( ( usAddress >= REG_HOLDING_START ) &&
        ( usAddress + usNRegs <= REG_HOLDING_START + REG_HOLDING_NREGS ) )
    {
        iRegIndex = ( int )( usAddress - usRegHoldingStart );
        switch ( eMode )
        {
            /* Pass current register values to the protocol stack. */
        case MB_REG_READ:
            while( usNRegs > 0 )
            {
                *pucRegBuffer++ = ( UCHAR ) ( usRegHoldingBuf[iRegIndex] >> 8 );
                *pucRegBuffer++ = ( UCHAR ) ( usRegHoldingBuf[iRegIndex] & 0xFF );
                iRegIndex++;
                usNRegs--;
            }
            break;

            /* Update current register values with new values from the
             * protocol stack. */
        case MB_REG_WRITE:
            while( usNRegs > 0 )
            {
                usRegHoldingBuf[iRegIndex] = *pucRegBuffer++ << 8;
                usRegHoldingBuf[iRegIndex] |= *pucRegBuffer++;
                iRegIndex++;
                usNRegs--;
            }
        }
    }
    else
    {
        eStatus = MB_ENOREG;
    }
    return eStatus;
}

/* 
 * Following implementation is not actually checked.
 */

eMBErrorCode
eMBRegCoilsCB( UCHAR * pucRegBuffer, USHORT usAddress, USHORT usNCoils, eMBRegisterMode eMode )
{
    eMBErrorCode    eStatus = MB_ENOERR;
    int             iIntRegIndex;
    int                iIntBufNum;
    int                iIntBitNum;
    int                iExtRegIndex=0;
    int                iExtBufNum;
    int                iExtBitNum;
    UCHAR            ucTemp;
    if( ( usAddress >= REG_COIL_START )
        && ( usAddress + usNCoils <= REG_COIL_START + REG_COIL_NREGS ) )
    {
        iIntRegIndex = ( int )( usAddress - usRegCoilStart );

        while( usNCoils > 0 )
        {
            iIntBufNum=iIntRegIndex/8;
            iIntBitNum=iIntRegIndex%8;
            iExtBufNum=iExtRegIndex/8;
            iExtBitNum=iExtRegIndex%8;

            switch ( eMode )
            {
            case MB_REG_READ:
                // Read coils
                if(iExtBitNum==0){
                    pucRegBuffer[iExtBufNum]=0;
                }
                ucTemp=(usRegCoilBuf[iIntBufNum]>>iIntBitNum) & 1;
                pucRegBuffer[iExtBufNum]|=ucTemp<<iExtBitNum;
                break;

            case MB_REG_WRITE:
                // Write coils
                ucTemp=usRegCoilBuf[iIntBufNum]&(~(1<<iIntBitNum));
                ucTemp|=((pucRegBuffer[iExtBufNum]>>iExtBitNum) & 1)<<iIntBitNum;
                usRegCoilBuf[iIntBufNum]=ucTemp;
                break;
            }
            iIntRegIndex++;
            iExtRegIndex++;
            usNCoils--;

        }
    }
    else
    {
        eStatus = MB_ENOREG;
    }

    return eStatus;
}

eMBErrorCode
eMBRegDiscreteCB( UCHAR * pucRegBuffer, USHORT usAddress, USHORT usNDiscrete )
{
    eMBErrorCode    eStatus = MB_ENOERR;
    int             iIntRegIndex;
    int                iIntBufNum;
    int                iIntBitNum;
    int                iExtRegIndex=0;
    int                iExtBufNum;
    int                iExtBitNum;
    UCHAR            ucTemp;
    if( ( usAddress >= REG_DISC_START )
        && ( usAddress + usNDiscrete <= REG_DISC_START + REG_DISC_NREGS ) )
    {
        iIntRegIndex = ( int )( usAddress - usRegDiscStart );

        while( usNDiscrete > 0 )
        {
            iIntBufNum=iIntRegIndex/8;
            iIntBitNum=iIntRegIndex%8;
            iExtBufNum=iExtRegIndex/8;
            iExtBitNum=iExtRegIndex%8;

            // Read discrete inputs
            if(iExtBitNum==0){
                pucRegBuffer[iExtBufNum]=0;
            }
            ucTemp=(usRegDiscBuf[iIntBufNum]>>iIntBitNum) & 1;
            pucRegBuffer[iExtBufNum]|=ucTemp<<iExtBitNum;

            iIntRegIndex++;
            iExtRegIndex++;
            usNDiscrete--;

        }
    }
    else
    {
        eStatus = MB_ENOREG;
    }

    return eStatus;
}
