/**
* Example app for using the Cayenne MQTT C++ library to send and receive example data. This example uses
* the X-NUCLEO-IDW01M1 WiFi expansion board via the X_NUCLEO_IDW01M1v2 library.
*/

#include "MQTTTimer.h"
#include "CayenneMQTTClient.h"
#include "MQTTNetworkIDW01M1.h"
#include "SpwfInterface.h"

// WiFi network info.
char* ssid = "TP-LINK_9E4F";
char* wifiPassword = "81379101";

// Cayenne authentication info. This should be obtained from the Cayenne Dashboard.
char* username = "b07cc1e0-6466-11eb-8779-7d56e82df461";
char* password = "ff7ea6d1ba46ada50b48d8b414e97877977f639e";
char* clientID = "d0c83450-6468-11eb-b767-3f1a8f1211ba";

SpwfSAInterface interface(D8, D2); // TX, RX
MQTTNetwork<SpwfSAInterface> network(interface);
CayenneMQTT::MQTTClient<MQTTNetwork<SpwfSAInterface>, MQTTTimer> mqttClient(network, username, password, clientID);

DigitalOut alarm(PH_0); // P21 sterowanie alarmem
DigitalIn drzwi1(PC_10); // P56 Drzwi wejściowe
DigitalIn drzwi2(PC_11); // P57 Drzwi balkonowe

DigitalOut str1(PC_7); // P48 SterowanieIn1
DigitalOut str2(PA_7); // P49 SterowanieIn2
DigitalOut str3(PA_6); // P50 SterowanieIn3
DigitalOut str4(PA_5); // P51 SterowanieIn4

DigitalOut swt1(PB_5); // P43 SwiatloIn1
DigitalOut swt2(PB_4); // P44 SwiatloIn2
DigitalOut swt3(PB_10); // P45 SwiatloIn3
DigitalOut swt4(PA_8); // P46 SwiatloIn3

DigitalIn okno1(PC_9); // P34 okno1
DigitalIn okno2(PC_6); // P35 okno2
DigitalIn okno3(PB_2); // P36 okno3
DigitalIn okno4(PB_1); // P37 okno4

DigitalIn czjn1(PC_1); // P39 czujnik1
DigitalIn czjn2(PC_2); // P40 czujnik2
DigitalIn czjn3(PC_3); // P41 czujnik3
DigitalIn czjn4(PC_0); // P42 czujnik4

/**
* Print the message info.
* @param[in] message The message received from the Cayenne server.
*/
void outputMessage(CayenneMQTT::MessageData& message)
{
    switch (message.topic)  {
    case COMMAND_TOPIC:
        printf("topic=Command");
        break;
    case CONFIG_TOPIC:
        printf("topic=Config");
        break;
    default:
        printf("topic=%d", message.topic);
        break;
    }
    printf(" channel=%d", message.channel);
    if (message.clientID) {
        printf(" clientID=%s", message.clientID);
    }
    if (message.type) {
        printf(" type=%s", message.type);
    }
    for (size_t i = 0; i < message.valueCount; ++i) {
        if (message.getValue(i)) {
            printf(" value=%s", message.getValue(i));
        }
        if (message.getUnit(i)) {
            printf(" unit=%s", message.getUnit(i));
        }
    }
    if (message.id) {
        printf(" id=%s", message.id);
    }
    printf("\n");
}

/**
* Handle messages received from the Cayenne server.
* @param[in] message The message received from the Cayenne server. */

void messageArrived(CayenneMQTT::MessageData& message)
{
    int error = 0;
    outputMessage(message);

    if (message.topic == COMMAND_TOPIC) {
        switch(message.channel) {
        case 0: // wirtualny kanał nr 0
            alarm = atoi(message.getValue());
            if ((error = mqttClient.publishData(DATA_TOPIC, message.channel, NULL, NULL, message.getValue())) != CAYENNE_SUCCESS) {
                printf("Publish LED state failure, error: %d\n", error);
            }
            break;
            
            case 3: // wirtualny kanał nr 3
            str1 = atoi(message.getValue());
            if ((error = mqttClient.publishData(DATA_TOPIC, message.channel, NULL, NULL, message.getValue())) != CAYENNE_SUCCESS) {
                printf("Publish LED state failure, error: %d\n", error);
            }
            break;
            
            case 6:
            // Set the onboard LED state
            str2 = atoi(message.getValue());
            // Publish the updated LED state
            if ((error = mqttClient.publishData(DATA_TOPIC, message.channel, NULL, NULL, message.getValue())) != CAYENNE_SUCCESS) {
                printf("Publish LED state failure, error: %d\n", error);
            }
            break;
            
            case 9:
            // Set the onboard LED state
            str3 = atoi(message.getValue());
            // Publish the updated LED state
            if ((error = mqttClient.publishData(DATA_TOPIC, message.channel, NULL, NULL, message.getValue())) != CAYENNE_SUCCESS) {
                printf("Publish LED state failure, error: %d\n", error);
            }
            break;
            
            case 12:
            // Set the onboard LED state
            str4 = atoi(message.getValue());
            // Publish the updated LED state
            if ((error = mqttClient.publishData(DATA_TOPIC, message.channel, NULL, NULL, message.getValue())) != CAYENNE_SUCCESS) {
                printf("Publish LED state failure, error: %d\n", error);
            }
            break;
            
            case 15:
            // Set the onboard LED state
            swt1 = atoi(message.getValue());
            // Publish the updated LED state
            if ((error = mqttClient.publishData(DATA_TOPIC, message.channel, NULL, NULL, message.getValue())) != CAYENNE_SUCCESS) {
                printf("Publish LED state failure, error: %d\n", error);
            }
            break;
            
            case 16:
            // Set the onboard LED state
            swt2 = atoi(message.getValue());
            // Publish the updated LED state
            if ((error = mqttClient.publishData(DATA_TOPIC, message.channel, NULL, NULL, message.getValue())) != CAYENNE_SUCCESS) {
                printf("Publish LED state failure, error: %d\n", error);
            }
            break;
            
            case 17:
            // Set the onboard LED state
            swt3 = atoi(message.getValue());
            // Publish the updated LED state
            if ((error = mqttClient.publishData(DATA_TOPIC, message.channel, NULL, NULL, message.getValue())) != CAYENNE_SUCCESS) {
                printf("Publish LED state failure, error: %d\n", error);
            }
            break;
            
            case 18:
            // Set the onboard LED state
            swt4 = atoi(message.getValue());
            // Publish the updated LED state
            if ((error = mqttClient.publishData(DATA_TOPIC, message.channel, NULL, NULL, message.getValue())) != CAYENNE_SUCCESS) {
                printf("Publish LED state failure, error: %d\n", error);
            }
            break;
        }
        
        // If this is a command message we publish a response. Here we are just sending a default 'OK' response.
        // An error response should be sent if there are issues processing the message.
        if ((error = mqttClient.publishResponse(message.id, NULL, message.clientID)) != CAYENNE_SUCCESS) {
            printf("Response failure, error: %d\n", error);
        }
    }
}

/**
* Connect to the Cayenne server.
* @return Returns CAYENNE_SUCCESS if the connection succeeds, or an error code otherwise.
*/
int connectClient(void)
{
    int error = 0;
    // Connect to the server.
    printf("Connecting to %s:%d\n", CAYENNE_DOMAIN, CAYENNE_PORT);
    while ((error = network.connect(CAYENNE_DOMAIN, CAYENNE_PORT)) != 0) {
        printf("TCP connect failed, error: %d\n", error);
        wait(2);
    }

    if ((error = mqttClient.connect()) != MQTT::SUCCESS) {
        printf("MQTT connect failed, error: %d\n", error);
        return error;
    }
    printf("Connected\n");

    // Subscribe to required topics.
    if ((error = mqttClient.subscribe(COMMAND_TOPIC, CAYENNE_ALL_CHANNELS)) != CAYENNE_SUCCESS) {
        printf("Subscription to Command topic failed, error: %d\n", error);
    }
    if ((error = mqttClient.subscribe(CONFIG_TOPIC, CAYENNE_ALL_CHANNELS)) != CAYENNE_SUCCESS) {
        printf("Subscription to Config topic failed, error:%d\n", error);
    }

    // Send device info. Here we just send some example values for the system info. These should be changed to use actual system data, or removed if not needed.
    mqttClient.publishData(SYS_VERSION_TOPIC, CAYENNE_NO_CHANNEL, NULL, NULL, CAYENNE_VERSION);
    mqttClient.publishData(SYS_MODEL_TOPIC, CAYENNE_NO_CHANNEL, NULL, NULL, "mbedDevice");
    //mqttClient.publishData(SYS_CPU_MODEL_TOPIC, CAYENNE_NO_CHANNEL, NULL, NULL, "CPU Model");
    //mqttClient.publishData(SYS_CPU_SPEED_TOPIC, CAYENNE_NO_CHANNEL, NULL, NULL, "1000000000");

    return CAYENNE_SUCCESS;
}

/**
* Main loop where MQTT code is run.
*/
void loop(void)
{
    // Start the countdown timer for publishing data every 5 seconds. Change the timeout parameter to publish at a different interval.
    MQTTTimer timer(2000);

    while (true) {
        // Yield to allow MQTT message processing.
        mqttClient.yield(1000);

        // Check that we are still connected, if not, reconnect.
        if (!network.connected() || !mqttClient.connected()) {
            network.disconnect();
            mqttClient.disconnect();
            printf("Reconnecting\n");
            while (connectClient() != CAYENNE_SUCCESS) {
                wait(2);
                printf("Reconnect failed, retrying\n");
            }
        }

        // Publish some example data every few seconds. This should be changed to send your actual data to Cayenne.
        if (timer.expired()) {
            int error = 0;
            if ((error = mqttClient.publishData(DATA_TOPIC, 1, NULL, NULL, drzwi1)) != CAYENNE_SUCCESS) // wartośc 1 oznacza nr wirtualnego kanału
            {
                printf("Publish temperature failed, error: %d\n", error);
            }
            if ((error = mqttClient.publishData(DATA_TOPIC, 2, NULL, NULL, drzwi2)) != CAYENNE_SUCCESS) // wartośc 2 oznacza nr wirtualnego kanału
            {
                printf("Publish temperature failed, error: %d\n", error);
            }
            if ((error = mqttClient.publishData(DATA_TOPIC, 4, NULL, NULL, czjn1)) != CAYENNE_SUCCESS) // wartośc 4 oznacza nr wirtualnego kanału
            {
                printf("Publish temperature failed, error: %d\n", error);
            }
            if ((error = mqttClient.publishData(DATA_TOPIC, 7, NULL, NULL, czjn2)) != CAYENNE_SUCCESS) 
            {
                printf("Publish temperature failed, error: %d\n", error);
            }
            if ((error = mqttClient.publishData(DATA_TOPIC, 10, NULL, NULL, czjn3)) != CAYENNE_SUCCESS) 
            {
                printf("Publish temperature failed, error: %d\n", error);
            }
            if ((error = mqttClient.publishData(DATA_TOPIC, 13, NULL, NULL, czjn4)) != CAYENNE_SUCCESS) 
            {
                printf("Publish temperature failed, error: %d\n", error);    
            }
            if ((error = mqttClient.publishData(DATA_TOPIC, 5, NULL, NULL, okno1)) != CAYENNE_SUCCESS) 
            {
                printf("Publish temperature failed, error: %d\n", error);
            }
            if ((error = mqttClient.publishData(DATA_TOPIC, 8, NULL, NULL, okno2)) != CAYENNE_SUCCESS) 
            {
                printf("Publish temperature failed, error: %d\n", error);
            }
            if ((error = mqttClient.publishData(DATA_TOPIC, 11, NULL, NULL, okno3)) != CAYENNE_SUCCESS) 
            {
                printf("Publish temperature failed, error: %d\n", error);
            }
            if ((error = mqttClient.publishData(DATA_TOPIC, 14, NULL, NULL, okno4)) != CAYENNE_SUCCESS) 
            {
                printf("Publish temperature failed, error: %d\n", error);
            }
            // Restart the countdown timer for publishing data every 5 seconds. Change the timeout parameter to publish at a different interval.
            timer.countdown_ms(2000);
        }
    }
}

/**
* Main function.
*/
int main()
{   
    // Initialize the network interface.
    printf("Initializing interface\n");
    interface.connect(ssid, wifiPassword, NSAPI_SECURITY_WPA2);

    // Set the default function that receives Cayenne messages.
    mqttClient.setDefaultMessageHandler(messageArrived);

    // Connect to Cayenne.
    if (connectClient() == CAYENNE_SUCCESS) {
        // Run main loop.
        loop();
    }
    else {
        printf("Connection failed, exiting\n");
    }

    if (mqttClient.connected())
        mqttClient.disconnect();
    if (network.connected())
        network.disconnect();

    return 0;
}

