/**
* @author Giles Barton-Owen
*
* @section LICENSE
*
* Copyright (c) 2011 mbed
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*
* @section DESCRIPTION
*    PWM/Duty cycle measurement library using Timer2: library h file for NXP LPC1768
*
*/ 

#ifndef PWMAVERAGE_H
#define PWMAVERAGE_H
#include "mbed.h"


/** A class for measuring PWM/Duty cycle of a signal connected to pins 30 and 29
 *
<<<<<<< local
 * Example:
=======
 *
>>>>>>> other
 * @code
 * // Measure and print the average duty cycle of a signal connected to p29 and p30 (together) while the button (p16) is pulled high
 *
 * #include "mbed.h"
 * #include "PWMAverage.h"
 *
 * DigitalOut myled(LED1);
 *
 * PWMAverage pa(p29,p30);
 * 
 * DigitalIn button (p16);
 * 
 * Timer tmr;
 * 
 * int main()
 * {
 *     button.mode(PullDown);
 *     while(1)
 *     {
 *         pa.reset();
 *     
 *         while (!button) {}
 *         pa.start();
 *         tmr.start();
 *         myled=1;
 *     
 *         while (button) {}
 *         pa.stop();
 *         tmr.stop();
 *         myled=0;
 *     
 *         printf("Average dudy cycle over %d us was %.4f\n\r",tmr.read_us(),pa.read());
 *     }
 * }
 * @endcode
 */
class PWMAverage
{
    public:
    /** Create a PWMAverage object
     * 
     * @param cap0 Pin connected to signal (ignored in program)
     * @param cap1 Pin connected to signal (ignored in program)
     */
    PWMAverage(PinName cap0, PinName cap1);
    
    /** Reset the counters and values
     */
    void reset();
    /** Start the timer
     */
    void start();
    /** Stop the timer
     */
    void stop();
    /** Read the duty cycle measured
     * @returns The Duty cycle
     */
    float read();
    /** Read the average length of time the signal was high per cycle in seconds
     * @returns The Length of time the signal was high per cycle in seconds
     */
    double avg_up();
    /** Read the average length of time the signal was high per cycle in seconds
     * @returns The Length of time the signal was high per cycle in seconds
     */
    float avg_UP();
    /** Read the average length of time the signal was low per cycle in seconds
     * @returns The Length of time the signal was low per cycle in seconds
     */
    double avg_down();
    /** Read the average period in seconds
     * @returns The Period
     */
    double period();
    /** Read the number of cycles counted over
     * @returns The number of cycles
     */
    int count();
    
    private:
    
    static void _tisr();
    static PWMAverage * instance;
    void enable(bool yn);
    void configure();
    
    long count_;
    long totalup;
    long total;
    
    double timeMult;
    
    double timeDiv;
    
    bool running;
    bool starting;
    
    uint32_t prescaler_point;
    
};


#endif