/* Small library to debug and alert with a single LED
 * Copyright (c) 2011 p07gbar
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 */

#ifndef FLASHER_H
#define FLASHER_H

#include "mbed.h"

#define NUMSTATES 5
#define OFF 0
#define ON 1
#define SLOW 2
#define MEDIUM 3
#define QUICK 4
#define FAST 4

/** A LED Flasher class based on Ticker and Timeout
 *  
 * States are defined in words:
 * 
 * OFF, ON, SLOW, MEDIUM, QUICK or FAST
 * 
 * With default times of (in seconds):
 * 
 * Slow: 2
 * 
 * Medium: 1
 * 
 * Fast/Quick: 0.3
 * 
 *
 * Example:
 * @code
 * // Switches LED1 through the different modes
 * #include "mbed.h"
 * #include "Flasher.h"
 * 
 * Flasher myFlasher(LED1, OFF); //Defines myFlasher on LED1 starting off
 * 
 * int main() {
 *      while(1){   
 *          wait(5);
 *          myFlasher = ON;
 *          wait(5);
 *          myFlasher.updateFlash(SLOW);
 *          wait(5);
 *          myFlasher = MEDIUM;
 *          wait(5);
 *          myFlasher.updateFlash(QUICK);
 *          wait(5);
 *          myFlasher.onFor(3,ON,MEDIUM);
 *          wait(5);
 *          myFlasher.pauseFor(2);
 *          wait(5);
 *          myFlasher.updateFlash(OFF);
 *      }
 *  }
 * @endcode
 */


class Flasher {

public:

/** Create a Flasher object attached to the specified Pin and Start State
 *
 * @param pin DigitalOut pin to connect the LED to
 * @param startState State for the object to start in
 *
 */
Flasher(PinName pin, int startState);

/** Set new state for the obeject
 *
 * @param statein New State for the object
 */
 
void updateFlash(int statein);

/** Read the current state of the object
 * 
 * @param returns The current state
 */
 
int getState();

/** Turn the LED off for a specified period, returns to old state
 * 
 * @param time Time in seconds to be off for
 */

void pauseFor(float time);

/** Sets a new state for a while then returns to the return state
 * 
 * @param time Time in seconds to be in new state for
 * @param newState The new state for it to be in
 * @param returnState The state for it to return to
 */

void onFor(float time, int newState, int returnState);

/** Stops the flasher until resume is called
 */

void pause();

/** Resumes the flasher
 */
 
void resume();

/** Configures the flash times for a specified state
 * 
 * @param statein State to change
 * @param newTime new flash period
 */
 
void setFlashTime(int statein, float newTime);

/** Reads back the flash time for a state
 * 
 * @param statein State to read back
 * @param returns Time of that states flash
 */

float getFlashTime(int statein);

/** Sets state quickly
 * 
 * @param statein State to change to
 */

Flasher& operator= (int statein);

/** Gets State quickly
 *
 * @param returns Current state
 */

operator int();

private:

DigitalOut _led;

int state;

int returntoState;

bool active;

float forTime;

float flashTimes[NUMSTATES];

Ticker flashbase;

Timeout forbase;

void tick();

void timeout();

int checkState(int statein);

};

#endif



