/*
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

#ifndef MICROBIT_MPR121_H
#define MICROBIT_MPR121_H

#include "mbed.h"
#include "MicroBitConfig.h"
#include "MicroBitPin.h"
#include "MicroBitComponent.h"
#include "MicroBitI2C.h"

/**
  * Relevant pin assignments
  */
#define MPR121_DEFAULT_INT          MICROBIT_PIN_P16

/**
  * I2C constants
  */
#define MPR121_DEFAULT_ADDR    0x5A


/**
  * Status Bits
  */


/**
  * MPR121 MAGIC ID value
  * Returned from the MPR121_WHO_AM_I register for ID purposes.
  */
#define MPR121_WHOAMI_VAL 0X9A //TODO

/**
  * Class definition for MicroBit Mpr121.
  *
  * Represents an implementation of the Mpr121.
  * Also includes ...
  */
class MicroBitMpr121 : public MicroBitComponent
{
    uint16_t                address;                  // I2C address of the magnetmometer.

    DigitalIn               int1;                     // Data ready interrupt.
    MicroBitI2C&            i2c;                      // The I2C interface the sensor is connected to.

    volatile uint16_t       _button;
    volatile uint32_t       _button_has_changed;

    /** The interrupt handler for the IRQ pin
    */
    void handler(void);

public:

    /**
     *  @enum MPR121_ADDR
     *  @brief Possible terminations for the ADDR pin
     */
    enum MPR121_ADDR {
        ADDR_VSS = 0x5A, /*!< ADDR connected to VSS */
        ADDR_VDD,    /*!< ADDR connected to VDD */
        ADDR_SCL,    /*!< ADDR connected to SDA */
        ADDR_SDA     /*!< ADDR connected to SCL */
    };

    /**
    *  @enum MPR121_REGISTER
    *  @brief The device register map
    */
    enum MPR121_REGISTER {
        ELE0_7_STAT = 0x00,
        ELE8_11_STAT, ELE0_7_OOR_STAT, ELE8_11_OOR_STAT, EFD0LB, EFD0HB,
        EFD1LB, EFD1HB, EFD2LB, EFD2HB, EFD3LB, EFD3HB, EFD4LB, EFD4HB, EFD5LB, EFD5HB,

        EFD6LB = 0x10,
        EFD6HB, EFD7LB, EFD7HB, EFD8LB, EFD8HB, EFD9LB, EFD9HB, EFD10LB,
        EFD10HB, EFD11LB, EFD11HB, EFDPROXLB, EFDPROXHB, E0BV, E1BV,

        E2BV = 0x20,
        E3BV, E4BV, E5BV, E6BV, E7BV, E8BV, E9BV, E10BV, E11BV, EPROXBV,
        MHDR, NHDR, NCLR, FDLR, MHDF,

        NHDF = 0x30,
        NCLF, FDLF, NHDT, NCLT, FDLT, MHDPROXR, NHDPROXR, NCLPROXR,
        FDLPROXR, MHDPROXF, NHDPROXF, NCLPROXF, FDLPROXF, NHDPROXT, NCLPROXT,

        FDLPROXT = 0x40,
        E0TTH, E0RTH, E1TTH, E1RTH, E2TTH, E2RTH, E3TTH, E3RTH,
        E4TTH, E4RTH, E5TTH, E5RTH, E6TTH, E6RTH, E7TTH,

        E7RTH = 0x50,
        E8TTH, E8RTH, E9TTH, E9RTH, E10TTH, E10RTH, E11TTH, E11RTH,
        EPROXTTH, EPROXRTH, DT_DR, CDC_CONFIG, CDT_CONFIG, ECR, CDC0,

        CDC1 = 0x60,
        CDC2, CDC3, CDC4, CDC5, CDC6, CDC7, CDC8, CDC9, CDC10, CDC11, CDCPROX, CDT0_CDT1,
        CDT2_CDT3, CDT4_CDT5, CDT6_CDT7,

        CDT8_CDT9 = 0x70,
        CDT10_CDT11, CDTPROX, GPIO_CTRL0, GPIO_CTRL1, GPIO_DATA, GPIO_DIR, GPIO_EN, GPIO_SET,
        GPIO_CLR, GPIO_TOGGLE, AUTO_CFG0, AUTO_CFG1, USL, LSL, TL,

        SRST = 0x80
    };

    /**
    * Constructor.
    * Create a software representation of a mpr121.
    *
    * @param _i2c an instance of i2c, which the mpr121 is accessible from.
    *
    * @param address the address for the mpr121 register on the i2c bus. Defaults to MPR121_DEFAULT_ADDR.
    *
    * @param id the ID of the new MicroBitMpr121 object. Defaults to MPR121_DEFAULT_ADDR.
    *
    * @code
    * MicroBitI2C i2c(I2C_SDA0, I2C_SCL0);
    *
    * MicroBitMpr121 mpr121(i2c);
    * @endcode
    */
    MicroBitMpr121(MicroBitI2C& _i2c, uint16_t address = MPR121_DEFAULT_ADDR, DigitalIn interupt = MICROBIT_PIN_P16, uint16_t id = MPR121_DEFAULT_ADDR);

    /**
      * Attempts to read the 8 bit ID for validation purposes. TODO IF RELEVANT
      *
      * @return the 8 bit ID, or MICROBIT_I2C_ERROR if the request fails.
      *
      * @code
      * mrp121.whoAmI();
      * @endcode
      */
    int whoAmI();

    /**
      * Allow the IC to run and collect user input
      */
    void enable(void);
    
        /** print the register map and values to the console
     */
    void registerDump();

    /**
      * Stop the IC and put into low power mode
      */
    void disable(void);

    /**
      * Periodic callback from MicroBit idle thread.
      *
      * Calls updateSample().
      */
    virtual void idleTick();


    /**
      * Destructor for MicroBitMpr121, where we deregister this instance from the array of fiber components.
      */
    ~MicroBitMpr121();

private:

    /**
      * Issues a standard, 2 byte I2C command write to the accelerometer.
      *
      * Blocks the calling thread until complete.
      *
      * @param reg The address of the register to write to.
      *
      * @param value The value to write.
      *
      * @return MICROBIT_OK on success, MICROBIT_I2C_ERROR if the the write request failed.
      */
    int writeCommand(uint8_t reg, uint8_t value);

    /**
      * Issues a read command, copying data into the specified buffer.
      *
      * Blocks the calling thread until complete.
      *
      * @param reg The address of the register to access.
      *
      * @param buffer Memory area to read the data into.
      *
      * @param length The number of bytes to read.
      *
      * @return MICROBIT_OK on success, MICROBIT_INVALID_PARAMETER or MICROBIT_I2C_ERROR if the the read request failed.
      */
    int readCommand(uint8_t reg, uint8_t* buffer, int length);

    /**
      * Issues a read of a given address, and returns the value.
      *
      * Blocks the calling thread until complete.
      *
      * @param reg The address of the 16 bit register to access.
      *
      * @return The register value, interpreted as a 16 but signed value, or MICROBIT_I2C_ERROR.
      */
    int read16(uint8_t reg);

    /**
      * Issues a read of a given address, and returns the value.
      *
      * Blocks the calling thread until complete.
      *
      * @param reg The address of the 16 bit register to access.
      *
      * @return The register value, interpreted as a 8 bit unsigned value, or MICROBIT_I2C_ERROR if the read request failed.
      */
    int read8(uint8_t reg);

    /**
      * An initialisation member function.
      *
      * @param id the unique identifier for this instance.
      *
      * @param address the base address on the i2c bus.
      *
      * @param intPin the pin used for interrupt.
      */
    void init(uint16_t id, uint16_t address);
};

#endif