/*
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

#include "MicroBitConfig.h"
#include "MicroBitPin.h"
#include "MicroBitMpr121.h"
#include "MicroBitFiber.h"
#include "ErrorNo.h"

void MicroBitMpr121::init(uint16_t id, uint16_t address)
{
    this->id = id;
    this->address = address;

    // setup and registers - start with POR values (must be in stop mode)
    MicroBitMpr121::writeCommand(SRST, 0x63); //REG 0x80

    // Baseline Filtering Control Register (changes response sensitivity)
    // http://cache.freescale.com/files/sensors/doc/app_note/AN3891.pdf
    MicroBitMpr121::writeCommand(MHDR, 0x1);  //REG 0x2B
    MicroBitMpr121::writeCommand(NHDR, 0x1);  //REG 0x2C
    MicroBitMpr121::writeCommand(NCLR, 0x0);  //REG 0x2D
    MicroBitMpr121::writeCommand(FDLR, 0x0);  //REG 0x2E
    MicroBitMpr121::writeCommand(MHDF, 0x1);  //REG 0x2F
    MicroBitMpr121::writeCommand(NHDF, 0x1);  //REG 0x30
    MicroBitMpr121::writeCommand(NCLF, 0xFF); //REG 0x31
    MicroBitMpr121::writeCommand(FDLF, 0x2);  //REG 0x32

    // Touch / Release Threshold
    // cache.freescale.com/files/sensors/doc/app_note/AN3892.pdf
    for(int i=0; i<(12*2); i+=2) { // touch
        MicroBitMpr121::writeCommand(static_cast<MPR121_REGISTER>(E0TTH+i), 0x20); //REG 0x41...0x58 odd
    }
    for(int i=0; i<(12*2); i+=2) { // release
        MicroBitMpr121::writeCommand(static_cast<MPR121_REGISTER>(E0RTH+i), 0x10); //REG 0x41...0x58 even
    }

    // Debounce Register DR=b6...4, DT=b2...0
    MicroBitMpr121::writeCommand(DT_DR, 0x11); //REG 0x5B

    // Filter and Global CDC CDT Configuration (sample time, charge current)
    MicroBitMpr121::writeCommand(CDC_CONFIG, 0x10); //REG 0x5C default 10
    MicroBitMpr121::writeCommand(CDT_CONFIG, 0x20); //REG 0x5D default 24

    // Auto-Configuration Registers
    // http://cache.freescale.com/files/sensors/doc/app_note/AN3889.pdf
    MicroBitMpr121::writeCommand(AUTO_CFG0, 0x33); // REG 0x7B
    MicroBitMpr121::writeCommand(AUTO_CFG1, 0x07); // REG 0x7C
    MicroBitMpr121::writeCommand(USL, 0xc9);       // REG 0x7D((3.3-.07)/3.3) * 256
    MicroBitMpr121::writeCommand(LSL, 0x83);       // REG 0x7E((3.3-.07)/3.3) * 256 * 0.65f
    MicroBitMpr121::writeCommand(TL,  0xb5);       // REG 0x7F((3.3-.07)/3.3) * 256 * 0.9f
    // 255 > USL > TL > LSL > 0

    // Electrode Configuration Register - enable all 12 and start
    MicroBitMpr121::writeCommand(ECR, 0x8f);

    // Indicate that we're up and running.
    status |= MICROBIT_COMPONENT_RUNNING;
}

MicroBitMpr121::MicroBitMpr121(MicroBitI2C& _i2c, uint16_t address, DigitalIn intPin, uint16_t id) :
    int1(intPin), // Not sure what this does :( and is our pin needs to be pull up !!!!!!!!!!!!!!
    i2c(_i2c)

{
    init(id, address);
}

void  MicroBitMpr121::enable(void)
{
    _button = 0;
    _button_has_changed = 0;
    // enable the 12 electrodes - allow disable to put device into
    //  lower current consumption mode
    MicroBitMpr121::writeCommand(ECR, 0x8f);
    // and attach the interrupt handler TODO
    // _irq->fall(this, &MicroBitMpr121::handler);

    return;
}

void MicroBitMpr121::registerDump()
{
    for(uint8_t i=0; i<0x80; i++) {
        int response = read8(i);
        printf("Reg 0x%02x: 0x%02x \n", i, response);
    }
    return;
}

void  MicroBitMpr121::disable(void)
{
    // detach the interrupt handler TODO
    //_irq->fall(NULL);
    _button = 0;
    _button_has_changed = 0;
    //  put the device in low current consumption mode - dont re-init registers
    MicroBitMpr121::writeCommand(ECR, 0x0);
    MicroBitMpr121::writeCommand(AUTO_CFG0, 0x0); // REG 0x7B
    MicroBitMpr121::writeCommand(AUTO_CFG1, 0x0); // REG 0x7C

    return;
}

int MicroBitMpr121::writeCommand(uint8_t reg, uint8_t value)
{
    uint8_t command[2];
    command[0] = reg;
    command[1] = value;

    return i2c.write(address, (const char *)command, 2);
}

int MicroBitMpr121::readCommand(uint8_t reg, uint8_t* buffer, int length)
{
    int result;

    if (buffer == NULL || length <= 0)
        return MICROBIT_INVALID_PARAMETER;

    result = i2c.write(address, (const char *)&reg, 1, true);
    if (result !=0)
        return MICROBIT_I2C_ERROR;

    result = i2c.read(address, (char *)buffer, length);
    if (result !=0)
        return MICROBIT_I2C_ERROR;

    return MICROBIT_OK;
}

int MicroBitMpr121::read16(uint8_t reg)
{
    uint8_t cmd[2];
    int result;

    cmd[0] = reg;
    result = i2c.write(address, (const char *)cmd, 1);
    if (result !=0)
        return MICROBIT_I2C_ERROR;

    cmd[0] = 0x00;
    cmd[1] = 0x00;

    result = i2c.read(address, (char *)cmd, 2);
    if (result !=0)
        return MICROBIT_I2C_ERROR;

    return (int16_t) ((cmd[1] | (cmd[0] << 8))); //concatenate the MSB and LSB
}

int MicroBitMpr121::read8(uint8_t reg)
{
    uint8_t data;
    int result;

    data = 0;
    result = readCommand(reg, (uint8_t*) &data, 1);
    if (result != MICROBIT_OK)
        return MICROBIT_I2C_ERROR;

    return data;
}

void MicroBitMpr121::idleTick()
{

}

int MicroBitMpr121::whoAmI()
{
    uint8_t data;
    int result;

//   result = readCommand(MPR121_WHOAMI, &data, 1);
    if (result != MICROBIT_OK)
        return MICROBIT_I2C_ERROR;

    return (int)data;
}

MicroBitMpr121::~MicroBitMpr121()
{
    fiber_remove_idle_component(this);
}
