#ifndef XBEE900HP_H
#define XBEE900HP_H

#include "mbed.h"

/**
 * Wrapper for the xBee 900 HP Module
 */
class xbee900hp
{
public:

    /**
    * Initialize the xBee Module
    *
    * @param pin_mosi -> mbed SPI MOSI pin
    * @param pin_miso -> mbed SPI MISO pin
    * @param pin_sck  -> mbed SPI SCK pin
    * @param pin_attn -> Signals that some message from the XBee module is available
    * @param pin_rst  -> Resets the XBee module
    * @param freq     -> sets the SPI clock frequency. The XBee has a maximum SPI clock rate of 3.5 MHz.
    */
    xbee900hp(PinName pin_mosi,PinName pin_miso,PinName pin_sck,PinName pin_attn, PinName pin_rst, unsigned int freq);

    /**
    * Destructor
    */
    ~xbee900hp();

    /**
    * Subroutine to reset the xBee Module
    */
    void reset();

    /**
    * Send packet over spi to xBee for TX
    *
    * @param data      -> char array of the message to be sent wirelessly.
    * @param length    -> then length of the char array excluding any null termination.
    * @param enMesh    -> true enables DigiMesh.
    *
    * Returns 0 upon successful completion, else returns 1.
    */
    int sendPacket(char* data, unsigned int length, bool enMesh);

    /**
    * Get signal strength of last recieved packet
    *
    * Returns RSSI in -dBm units (0x00-0xFF), else returns 1 indicating failure.
    */
    unsigned int getRSSI();

    /**
    * Set the transmission power level.
    * @param value integer value of power level. Acceptable values are 0-4.
    *
    * @returns 
    *   0 upon success
    *   1 upon failure
    */
    /* Note
    * 0 = +7 dBm, (5 mW)
    * 1 = +15 dBm, (32 mW)
    * 2 = +18 dBm, (63 mW)
    * 3 = +21 dBm, (125 mW)
    * 4 = +24 dBm, (250 mW)
    */
    int setPower(int value);

    /**
    * Get serial number of XBee module
    *
    * Returns the XBee's unique 64-bit (0-0xFFFFFFFF) Serial/MAC address, else returns 1.
    */
    unsigned long long getSerial();

    /**
    * Wait for and read incoming data packet
    *
    * @param data   -> A char array of sufficent size to contain any message you expect to recieve.
    *
    * Returns the 64-bit (0-0xFFFFFFFF) Serial/MAC address of the sending XBee if successful, or 0 if it failed.
    */
    unsigned long long readPacket(char* data);

    /**
    * Clear output buffer
    */
    void clearBuff();

    /**
    * Check ATTN signal
    *
    * Returns the state of the ATTN pin.
    * NOTE: This pin is inverse logic. (i.e. 0 = message waiting, 1 = no messages availiable)
    */
    int attn();

    /**
    * Read raw data in from module
    *
    * Returns a single char from the XBee.
    * NOTE: Check the status of the XBee's ATTN pin to see if data is available first.
    */
    char xbee_getc();

private:
    // Setup pin input types.

    // Reset pin (Pin 5 on XBee)
    DigitalOut _pin_rst;
    // Attention Pin (Pin 19 on XBee
    DigitalIn _pin_attn;

    // Define SPI connection for the XBee
    SPI _xbeespi;

};

#endif
