#include "Parallax_X-Band.h"

/** 
* Initialize the xband motion detector and setup interrupts. Default is to be disabled.
*
* @param pin_enable Enable pin. Connected to EN on xband.
* @param pin_input Data pin. Connected to OUT on xband.
*/ 
xband::xband(PinName pin_enable, PinName pin_input)
    : _pin_enable(pin_enable), _pin_input(pin_input)
{
    // Default disabled.
    _pin_enable=0;
    
    // Initialize count to be 0.
    _count = 0;
    
    // No velocity reading has been calculated yet.
    _velocityflag = false;
    
    // Setup interrupt on input pin.
    _pin_input.rise(this, &xband::xband_RiseISR);
    
    // Timer for count differntiation.
    _sampletime.start();

    // Interrupt to calculate the velocity. Default every half second.
    _samplevelocity.attach(this, &xband::xband_velocityISR, 0.5);
}

/**
* Destructor
*/
xband::~xband() {}



/** 
* Subroutine to enable and disable the detector.
*
* @param enable True to enable, False to disable.
*/ 
void xband::enable(bool enable)
{
    _pin_enable=enable;
}

/** 
* Interrupt service routine to collect data on the input pin.
*/ 
void xband::xband_RiseISR()
{
    // Increment count for every positive rising edge pulse on input pin.
    _count++;
}

/** 
* Interrupt service routine to calculate the velocity of the moving object.
*/ 
void xband::xband_velocityISR()
{
    // Differentiate current count value change during time slice.
    _velocity = ((_count - _lastcount)/(_sampletime.read_ms()/1000.0));
    
    // Set a flag to indicate a new velocity calculation.
    _velocityflag = true;
    
    // Reset timer for next differentiation.
    _sampletime.reset();
    _lastcount = _count;
}

/** 
* Get count value and return it as an integer.
*
* @return Integer value of current count.
*/ 
int xband::read_count()
{
    return _count;
}

/** 
* Get current velocity reading in ticks/sec. Sets a flag to false to indicate a stale value after reading.
*
* @return Float value of currently measured velocity.
*/
float xband::read_velocity()
{
    _velocityflag = false;
    return _velocity;
}

/** 
* Resets the count variable to 0.
*/ 
void xband::reset_count()
{
    _count = 0;
}

/** 
* Checks if a new velocity value has been calculated.
*
* @return True if a new velocity value is avaliable. False if the value has
* already been read.
*/ 
bool xband::velocityack()
{
    return _velocityflag;
}

/** 
* Default overload to read velocity.
*
* @return Float value of currently measured velocity.
*/ 
xband::operator float() {
    return read_velocity();
}