

#include "mbed.h"
#include "NeoStrip.h"

// function to write to the strip, implemented in ARM assembly
extern "C" void neo_out(NeoColor*, int);

// FastIO register address and bitmask for the GPIO pin
// because these are imported in the assembly
uint32_t neo_fio_reg;
uint32_t neo_bitmask;

NeoStrip::NeoStrip(PinName pin, int N) : N(N)
{
	bright = 0.5;
	Nbytes = N * 3;
	strip = (NeoColor*)malloc(N * sizeof(NeoColor));
	if (strip == NULL)
	{
		printf("NeoStrip: ERROR unable to malloc strip data");
		N = 0;
	}
	gpio_init_out(&gpio, pin);
	//gpio_init(&gpio, pin, PIN_OUTPUT);		// initialize GPIO registers
	neo_fio_reg = (uint32_t)gpio.reg_dir;	// set registers and bitmask for
	neo_bitmask = 1 << ((int)pin & 0x1F);	// the assembly to use
}

void NeoStrip::setBrightness(float bright)
{
	this->bright = bright;
}

void NeoStrip::setPixel(int p, int color)
{
	int red = (color & 0xFF0000) >> 16;
	int green = (color & 0x00FF00) >> 8;
	int blue = (color & 0x0000FF);
	setPixel(p, red, green, blue);
}

void NeoStrip::setPixel(int p, uint8_t red, uint8_t green, uint8_t blue)
{
	// set the given pixel's RGB values
	// the array is indexed modulo N to avoid overflow
	// RDW: swapped g / r because who knows why it doesn't work
	strip[p % N].red = (uint8_t)(green * bright);
	strip[p % N].green = (uint8_t)(red * bright);
	strip[p % N].blue = (uint8_t)(blue * bright);
}

void NeoStrip::setPixels(int p, int n, const int *colors)
{
	int r, g, b;
	for (int i = 0; i < n; i++)
	{
		r = (colors[i] & 0xFF0000) >> 16;
		g = (colors[i] & 0x00FF00) >>8;
		b = colors[i] & 0x0000FF;
		setPixel(p+i, r, g, b);
	}
}

void NeoStrip::training(){
	for (int i = 0; i < 4; i++)
	{
		setPixel(i,(uint8_t)255,(uint8_t)165,(uint8_t)0);
	}
	write();
}

void NeoStrip::training_mode(int mode){
	for (int i = 0; i < 4; i++)
	{
		if(i==mode){
			setPixel(i,(uint8_t)0,(uint8_t)255,(uint8_t)0);
		}else{
			setPixel(i,(uint8_t)0,(uint8_t)0,(uint8_t)0);
		}
	}
	write();
}

void NeoStrip::clear()
{
	for (int i = 0; i < N; i++)
	{
		strip[i].red = 0;
		strip[i].green = 0;
		strip[i].blue = 0;
	}
	write();
}

void NeoStrip::write()
{
	__disable_irq();		// disable interrupts
	neo_out(strip, Nbytes);	// output to the strip
	__enable_irq();			// enable interrupts
	wait_us(50);			// wait 50us for the reset pulse
}

void NeoStrip::initialize()
{
	const int red[] = {0xFF0000, 0xFF0000, 0xFF0000, 0xFF0000};
    setPixels(0, 4, red);
    write();
}

void NeoStrip::progress(float time)
{
	for (int i = 0; i < 4; i++)
	{
		if(time >= (i+1)*0.25){
			setPixel(i, 0, 255, 0);
		}else{
			setPixel(i, 255, 0, 0);
		}
	}
	write();
}

void NeoStrip::start_up(){  
    for (int i = 0; i < 4; i++)
    {
        strip[i].red = 0;
        strip[i].green = 0;
        strip[i].blue = 255;
        write();
        wait(0.1);
        clear();
    }
    for (int i = 0; i < 4; i++)
    {
        strip[i].red = 0;
        strip[i].green = 0;
        strip[i].blue = 255;
        write();
        wait(0.15);
        clear();
    }
}
 
void NeoStrip::bad_breath() {
    const int red[] = {0xFF0000, 0xFF0000, 0xFF0000, 0xFF0000};
    setPixels(0, 4, red);
    write();
    wait(0.2);
    clear();
    wait(0.2);
    setPixels(0, 4, red);
    write();
    wait(0.2);
    clear();
    wait(0.2);
    setPixels(0, 4, red);
    write();
    wait(0.2);
    clear();
}
