/* Extended TextLCD library.
 *
 * Based on the mbed TextLCD Library, for a 4-bit LCD based on HD44780
 * Copyright (c) 2007-2010, sford, http://mbed.org
 *
 * Original license:
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef EXT_TEXT_LCD_TEXTLCD_H
#define EXT_TEXT_LCD_TEXTLCD_H

#include "mbed.h"

namespace ext_text_lcd {

/** A TextLCD interface for driving 4-bit HD44780-based LCDs
 *
 * Currently supports 16x2, 20x2 and 20x4 panels
 *
 * @code
 * #include "mbed.h"
 * #include "TextLCD.h"
 * 
 * TextLCD lcd(p10, p12, p15, p16, p29, p30); // rs, e, d4-d7 on individual pins
 * TextLCD lcd(p10, p12, Port2, 2); // rs, e, d4-d7 on P2.2-P2.5
 * 
 * int main() {
 *     lcd.printf("Hello World!\n");
 * }
 * @endcode
 */
class TextLCD : public Stream {
public:
    /** LCD panel format */
    class LCDType {
    public: 
        int columns() const;
        int rows() const;
        int address(int column, int row) const;
                
        typedef int (*MakeAddrFunc)(const LCDType &, int, int);
        
    private:
        LCDType(int columns, int rows, MakeAddrFunc makeAddr) : 
            _columns(columns), _rows(rows), _makeAddr(makeAddr)
        { }

        friend class TextLCD;
                
        int _columns;
        int _rows;
        MakeAddrFunc _makeAddr;
    };
    
    static const LCDType LCD16x2;
    static const LCDType LCD16x2B;
    static const LCDType LCD20x2;
    static const LCDType LCD20x4;

    /** Create a TextLCD interface
     *
     * @param rs    Instruction/data control line
     * @param e     Enable line (clock)
     * @param d4-d7 Data lines for using as a 4-bit interface
     * @param type  Sets the panel size/addressing mode (default = LCD16x2)
     */
    TextLCD(PinName rs, PinName e, PinName d4, PinName d5, PinName d6, PinName d7, const LCDType &type = LCD16x2);

    /** Create a TextLCD interface
     *
     * @param rs    Instruction/data control line
     * @param e     Enable line (clock)
     * @param dport GPIO port with all data lines
     * @param dlsb  Index of the lowest data bit in the GPIO port
     * @param type  Sets the panel size/addressing mode (default = LCD16x2)
     */
    TextLCD(PinName rs, PinName e, PortName dport, std::size_t dlsb, const LCDType &type = LCD16x2);
    
#if DOXYGEN_ONLY
    /** Write a character to the LCD
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formated string to the LCD
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formating the string.
     */
    int printf(const char* format, ...);
#endif

    /** Locate to a screen column and row
     *
     * @param column  The horizontal position from the left, indexed from 0
     * @param row     The vertical position from the top, indexed from 0
     */
    void locate(int column, int row);

    /** Clear the screen and locate to 0,0 */
    void cls();

    int columns() { return _type.columns(); }
    int rows() const { return _type.rows(); }

    enum DisplayControl {
        DisplayOn = 0x04,
        DisplayOff = 0
    };
    enum CursorDisplayControl {
        CursorOn = 0x02,
        CursorOff = 0
    };
    enum CursorStyle {
        BlinkingCursor = 0x01,
        UnderlineCursor = 0
    }; 
    void setDisplayControl(DisplayControl d = DisplayOn, CursorDisplayControl c = CursorOff, CursorStyle b = UnderlineCursor);

    void writeToCGMem(unsigned addr, const char *buffer, std::size_t size);

protected:
    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

    int address(int column, int row) const { return _type.address(column, row); }

    void writeByte(int value);
    void writeCommand(int command);
    void writeData(int data);

private:
    void init();
    
    struct LcdOutput {
        virtual void writeNibble(int value) = 0;
        virtual ~LcdOutput() { }
    };
    
    struct BusOutput : public LcdOutput {
        BusOutput(PinName d4, PinName d5, PinName d6, PinName d7) : _bus(d4, d5, d6, d7) { }
        virtual void writeNibble(int value);
    private:
        BusOut _bus;
    };
    
    struct PortOutput : public LcdOutput {
        PortOutput(PortName dport, std::size_t dlsb) : _port(dport, 0x0f << dlsb), _shift(dlsb) { }
        virtual void writeNibble(int value);
        
    private:
        PortOut _port;
        std::size_t _shift;
    };
    
    DigitalOut _rs, _e;
    LcdOutput *_output;
    LCDType _type;
    int _column;
    int _row;    
};

inline int TextLCD::LCDType::columns() const { 
    return this->_columns; 
}
inline int TextLCD::LCDType::rows() const { 
    return this->_rows; 
}
inline int TextLCD::LCDType::address(int column, int row) const { 
    return this->_makeAddr(*this, column, row); 
}

} // ext_text_lcd namespace

#endif
