/* mbed TextLCD Library, for a 4-bit LCD based on HD44780
 * Copyright (c) 2007-2010, sford, http://mbed.org
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "ext_text_lcd/TextLCD.h"
#include "mbed.h"

namespace ext_text_lcd {

namespace {
    int makeLcdAddress(const TextLCD::LCDType &type, int column, int row) {
        return column + row * 0x40;
    }
    
    int makeLcdAddress40(const TextLCD::LCDType &type, int column, int row) {
        return column + row * 40;
    }
    
    int makeLcdAddress4Rows(const TextLCD::LCDType &type, int column, int row) {
        return column + (row % 2) * 0x40 + (row / 2) * 20;
    }
}

const TextLCD::LCDType TextLCD::LCD16x2(16, 2, makeLcdAddress);
const TextLCD::LCDType TextLCD::LCD16x2B(16, 2, makeLcdAddress40);
const TextLCD::LCDType TextLCD::LCD20x2(20, 2, makeLcdAddress);
const TextLCD::LCDType TextLCD::LCD20x4(20, 4, makeLcdAddress4Rows);

TextLCD::TextLCD(PinName rs, PinName e, PinName d4, PinName d5,
                 PinName d6, PinName d7, const LCDType &type) : 
    _rs(rs), _e(e), _output(new BusOutput(d4, d5, d6, d7)),
    _type(type)
{
    init();
}

TextLCD::TextLCD(PinName rs, PinName e, PortName dport, std::size_t dlsb, 
                 const LCDType &type) :
    _rs(rs), _e(e), _output(new PortOutput(dport, dlsb)),
    _type(type)
{
    init();
}

void TextLCD::init() {
    _e  = 1;
    _rs = 0;            // command mode

    wait(0.015);        // Wait 15ms to ensure powered up

    // send "Display Settings" 3 times (Only top nibble of 0x30 as we've got 4-bit bus)
    for (int i=0; i<3; i++) {
        writeByte(0x3);
        wait(0.00164);  // this command takes 1.64ms, so wait for it
    }
    writeByte(0x2);     // 4-bit mode
    wait(0.000040f);    // most instructions take 40us

    writeCommand(0x28); // Function set 001 BW N F - -
    writeCommand(0x0C);
    writeCommand(0x6);  // Cursor Direction and Display Shift : 0000 01 CD S (CD 0-left, 1-right S(hift) 0-no, 1-yes
    cls();
}

void TextLCD::cls() {
    writeCommand(0x01); // cls, and set cursor to 0
    wait(0.00164f);     // This command takes 1.64 ms
    locate(0, 0);
}

void TextLCD::locate(int column, int row) {
    _column = column;
    _row = row;
    
    int a = address(column, row);
    writeCommand(0x80 | a);
}

int TextLCD::_putc(int value) {
    int column = _column;
    int row = _row;
    if (value == '\n') {
        column = 0;
        row = _row+1;
    } else {
        writeData(value);
        column++;
        if (column < columns()) {
            return value;
        }
        
        column = 0;
        row++;
    }
    if (row >= rows()) {
        row = 0;
    }
    locate(column, row);
    return value;
}

int TextLCD::_getc() {
    return -1;
}

void TextLCD::writeByte(int value) {
    _output->writeNibble(value >> 4);
    wait(0.000040f); // most instructions take 40us
    _e = 0;
    wait(0.000040f);
    _e = 1;
    _output->writeNibble(value);
    wait(0.000040f);
    _e = 0;
    wait(0.000040f);  // most instructions take 40us
    _e = 1;
}

void TextLCD::writeCommand(int command) {
    _rs = 0;
    writeByte(command);
}

void TextLCD::writeData(int data) {
    _rs = 1;
    writeByte(data);
}

void TextLCD::setDisplayControl(DisplayControl d, CursorDisplayControl c, CursorStyle b)
{
    writeCommand(0x08 | d | c | b);
}

void TextLCD::writeToCGMem(unsigned addr, const char *buffer, std::size_t size)
{
    writeCommand(0x40 | (addr & 0x3f));
    while (size > 0) {
        writeData(*buffer);
        size--;
        buffer++;
    }
    
    locate(_column, _row);
}

void TextLCD::BusOutput::writeNibble(int value) {
    _bus = value; 
}

void TextLCD::PortOutput::writeNibble(int value) {
    _port = value << _shift;
}

} // ext_text_lcd namespace
