#include "system_states.hpp"
#include "system.hpp"
#include "display_manager.hpp"

const char *statusMsg[] = {
    //234567890123456
    "Cmd: [5]Help",
    "Cmd: [0]Send",
    "Cmd: 1<-4-+-6->3",
    "Cmd: [7]BS[8]Del",
    "Cmd: [9]Clear",
    "Cmd: <*>Edit",
    NULL
};

enum {
    HelpTimeout = 2
};

void CommandState::enterState() {
    DisplayManager *display = system()->display();
    display->hideCursor();
    showStatusMsg(0);
    updateText();
}

void CommandState::exitState() {
    this->statusMsgTimeout.detach();
}

void CommandState::handleKey(char key)
{
    switch(key) {
    case '5':
        handleHelp(); break;
    case '1':
        handleHome(); break;
    case '3':
        handleEnd(); break;
    case '4':
        handleLeft(); break;
    case '6':
        handleRight(); break;
    case '7':
        handleBackSpace(); break;
    case '8':
        handleDelete(); break;
    case '9':
        handleClear(); break;
    case '0':
        handleSend(); break;
    case '@':
        system()->setState(System::Edit);
        break;
    default:
        break;
    }
}

void CommandState::handleHome() const {
    system()->moveCursorTo(0);
    updateCursor();
}

void CommandState::handleEnd() const {
    system()->moveCursorTo(-1);
    updateCursor();
}

void CommandState::handleLeft() const {
    system()->moveCursorBy(-1);
    updateCursor();
}

void CommandState::handleRight() const {
    system()->moveCursorBy(+1);
    updateCursor();
}

void CommandState::handleDelete() const {
    system()->deleteCurrentSymbol();
    updateText();
}

void CommandState::handleBackSpace() const {
    system()->moveCursorBy(-1);
    system()->deleteCurrentSymbol();
    updateText();
}

void CommandState::handleClear() const {
    system()->clearText();
    updateText();
}

void CommandState::handleSend() const {
    if (system()->text_size() > 0) {
        system()->setState(System::Sending);
    }
}

void CommandState::handleHelp() {
    this->statusMsgTimeout.detach();
    showNextStatusMsg();
}

void CommandState::showNextStatusMsg() {
    int msg = this->statusMsgIndex+1;
    if (statusMsg[msg] == NULL) {
        msg = 0;
    }
    showStatusMsg(msg);
}

void CommandState::showStatusMsg(int index) {
    this->statusMsgIndex = index;
    DisplayManager *display = system()->display();
    display->writeStatus(statusMsg[index]);
    this->statusMsgTimeout.attach(this, &CommandState::showNextStatusMsg, HelpTimeout * 1.0);    
}