#include "kbd_mgr/KeyboardMonitor.h"

namespace kbd_mgr {

KeyboardMonitor::KeyboardMonitor(PortName inPort, std::size_t numKeysPerRow, std::size_t inLowestBit, 
            const KeyboardMonitor::OutPinsSet &outPins) :
    in(inPort, ((1 << numKeysPerRow)-1) << inLowestBit), inBitShift(inLowestBit), outPins(outPins),
    ticker(), scanRow(0), currentState(outPins.size(), numKeysPerRow)
{ }

KeyboardMonitor::~KeyboardMonitor()
{
    stop();
}
    
void KeyboardMonitor::start(float pollingPeriod)
{   
    if (this->outPins.empty()) {
        return;
    }
    
    if (pollingPeriod < 20e-6) {
        pollingPeriod = 20e-6;
    }
    
    for(OutPinsSet::const_iterator p = this->outPins.begin(); p != this->outPins.end(); ++p) {
        DigitalOut out(*p);
        out.write( p == this->outPins.begin() ? 1 : 0 );
    }
    this->in.mode(PullDown);
    this->scanRow = 0; 
    this->ticker.attach(this, &KeyboardMonitor::timerHandler, pollingPeriod);
}

void KeyboardMonitor::stop()
{
    this->ticker.detach();
    this->in.mode(OpenDrain);
}

void KeyboardMonitor::timerHandler()
{
    DigitalOut out(this->outPins[this->scanRow]);
    out.write(1);
    wait_us(10);
    int v = (this->in.read() >> this->inBitShift);
    out.write(0);
    this->currentState.setRowState(this->scanRow, v);
    this->scanRow = (this->scanRow + 1) % this->currentState.getNumRows();
    if (this->scanRow == 0) {
        invokeHandler(this->currentState);
    }    
}

} // kbd_mgr
