#include "system.hpp"
#include "state.hpp"

#include "mbed.h"

#include <cstring>
#include <algorithm>
#include <iostream>

BusOut stateLeds(LED1, LED2);

System::System(DisplayManager *display, KeyboardManager *keyboard, DtmfGenerator *dtmf) : 
    display_(display), keyboard_(keyboard), dtmf_(dtmf), state_(NULL), text_(), len_(0), pos_(0)
{
    std::cout << "Init System" << "\r" << std::endl;     

    std::memset(this->states_, 0, sizeof(this->states_));
}

void System::registerState(StateId id, State *state)
{
    std::cout << "Register System State " << id << "\r" << std::endl;     
    this->states_[id] = state;
    if (this->state_ == NULL) {
        setState(id);
    }
}

void System::unregisterState(State *state)
{
    State **begin = &this->states_[0];
    State **end = begin + NumStates;
    State **p = std::find(begin, end, state);
    if (p != end) {
        *p = NULL;
    }
    
    if (this->state_ == state) {
        this->state_ = NULL;
    }
}

void System::setState(StateId newState)
{
    std::cout << "Set System state " << newState << "\r" << std::endl;     
    State *s = this->states_[newState];
    if (!s) {
        std::cout << "Target state not found !\r" << std::endl;
        return;
    }
    
    if (s != this->state_) {
        if (this->state_ != NULL) {
            std::cout << "Exit current state\r" << std::endl;
            this->state_->exitState();
        }
        
        stateLeds = 0;
        
        if (s != NULL) {
            std::cout << "Enter new state\r" << std::endl;
            s->enterState();
        }
        
        stateLeds = newState+1;
        
        this->state_ = s;
        std::cout << "State change complete\r" << std::endl;
    }
}

void System::insertSymbol(char ch)
{
    std::cout << "System Insert '" << ch << "'\r" << std::endl;     

    if (this->len_ == 19) {
        return;
    }
    
    if (this->pos_ == this->len_) {
        this->text_.append(1, ch);
    }
    else {
        this->text_.insert(this->pos_, 1, ch);
    }
    this->pos_++;
    this->len_++;
}

void System::deleteCurrentSymbol()
{
    std::cout << "System Delete" << "\r" << std::endl;     
    if (this->pos_ < this->len_) {
        this->text_.erase(this->pos_, 1);
        this->len_--;
        if (this->pos_ > this->len_) {
            this->pos_ = len_;
        }
    }
}

void System::clearText()
{
    std::cout << "System Clear" << "\r" << std::endl;     
    this->text_.clear();
    this->len_ = 0;
    this->pos_ = 0;
}

void System::moveCursorTo(int pos)
{
    std::cout << "System MoveTo " << pos << "/" << this->len_ << "\r" << std::endl;     
    if (pos < 0) {
        pos = this->len_+1 + pos;
    }
    setCursorPosition(pos);
}

void System::moveCursorBy(int delta)
{
    std::cout << "System MoveBy " << delta << "\r" << std::endl;     
    int pos = this->pos_ + delta;
    setCursorPosition(pos);
}

void System::setCursorPosition(int pos) 
{    
    std::cout << "System SetPos " << pos << "/" << this->len_ << "\r" << std::endl;     
    if (pos < 0) {
        pos = 0;
    }
    else if (pos > this->len_) {
        pos = this->len_;
    }
    this->pos_ = pos;
}

void System::handleKey(char ch)
{
    this->state()->handleKey(ch);
}
