#ifndef KEYBOARD_MONITOR_H_
#define KEYBOARD_MONITOR_H_

#include "kbd_mgr/KeyboardStateEventServer.h"
#include "kbd_mgr/KeyboardState.h"

#include "mbed.h"
#include <vector>

namespace kbd_mgr {

/**
 * @brief A class that polls the state of a switch-matrix keyboard.
 * For efficiency reasons, this class uses contiguous bits in a GPIO ports for reading the state of the keys in a row.
 * Key rows are activated one by one using a set of digital outputs.
 * When a full keyboard scan has been done, it is provided to the registered handler.
 */
class KeyboardMonitor : public KeyboardStateEventServer {
public:
    typedef std::vector<PinName> OutPinsSet;
    
    /**
     * @param inPort        Port to be used for reading the key state.
     * @param numKeysPerRow Number of keys in a row (N)
     * @param inLowestBit   Index of the lowest bit of inPort (using inLowestBit to inLowestBit+N).
     * @param outPins       Pins to be used for powering each key row.
     */
    KeyboardMonitor(PortName inPort, std::size_t numKeysPerRow, std::size_t inLowestBit, 
            const OutPinsSet &outPins);
    
    ~KeyboardMonitor();
    
    /**
     * @brief Starts the polling of the keyboard state.
     */
    void start(float pollingPeriod = 0.001);
    
    /**
     * @brief Disables the polling of the keyboard state.
     */
    void stop();
    
private:
    void timerHandler();

    PortIn in;
    int inBitShift;
    OutPinsSet outPins;

    Ticker ticker;
    std::size_t scanRow;            /*!< next key row to be scanned */
    KeyboardState currentState;     /*!< currently being built keyboard state */
};

} // kbd_mgr

#endif // KEYBOARD_MONITOR_H_
