#ifndef KEY_MAPPER_H_
#define KEY_MAPPER_H_

#include "kbd_mgr/KeyPressEventServer.h"

#include <vector>
#include <string>
#include <utility>

namespace kbd_mgr {

/**
 * @brief A class used to store a set of key mappings.
 * It associates a key event and a key code to a mapped character.
 * It also allows to specify a mapping for any unmatched key event of a key.
 * The KeyMap can be set at construction time with a string and an optional key event.
 * Additional mappings can be specified by using the () operator. This way, both individual mappings and string mappings are possible.
 * Eg. keymap(KeyEvent::LongKeyPress, 1, '@')(1, '*')
 *      maps the key 1 to '*' for all key presses but to '@' when a long key press occurs.
 */
class KeyMap {
public:
    /**
     * @brief Creates an empty keymap.
     */
    KeyMap() : map_() { }
    
    /**
     * @brief Creates a keymap based on a string.
     * Each character is mapped to the key at that position (key 1 maps to str[1]).
     */
    KeyMap(const std::string &str) : map_() {
        addMappings(KeyEvent::NoEvent, str);
    }
    /**
     * @brief Creates a keymap based on a string for a certain key event.
     * Each character is mapped to the key at that position (key 1 maps to str[1]).
     */
    KeyMap(KeyEvent::EventType event, const std::string &str) : map_() {
        addMappings(event, str);
    }
    
    /**
     * @brief Adds key mappings based on the given string.
     */
    KeyMap& operator()(const std::string &str) {
        addMappings(KeyEvent::NoEvent, str);
        return *this;
    }
    
    /**
     * @brief Adds key mappings based on the given string for a given event.
     */
    KeyMap& operator()(KeyEvent::EventType event, const std::string &str) {
        addMappings(event, str);
        return *this;
    }
    
    /**
     * @brief Adds a key mapping for a key.
     */
    KeyMap& operator()(int key, char ch) { 
        addMapping(KeyEvent::NoEvent, key, ch); 
        return *this; 
    }
    
    /**
     * @brief Adds a key mapping for a key event & code combo.
     */
    KeyMap& operator()(KeyEvent::EventType event, int key, char ch) { 
        addMapping(event, key, ch); 
        return *this; 
    }

    /**
     * @brief Gets the mapped character for a key press event.
     */    
    char map(KeyEvent::EventType event, int key) const;

private:
    struct KeyMapping {
        KeyEvent::EventType event;
        int key;
        char ch;
        
        KeyMapping(KeyEvent::EventType event, int key, char ch) : event(event), key(key), ch(ch) { }
        
        bool matches(const std::pair<KeyEvent::EventType, int> &arg) const {
            return this->event == arg.first && this->key == arg.second;
        }
    };
    
    void addMapping(KeyEvent::EventType event, int key, char ch);
    void addMappings(KeyEvent::EventType event, const std::string &str);
    const KeyMapping * getMapping(KeyEvent::EventType event, int key) const;
    
    typedef std::vector<KeyMapping> Map;
    Map map_;
};

/**
 * @brief A key press event handler that adds a mapped key char to key events.
 * Mappings can be specified for any key press event or for some specific key press event.
 */
class KeyMapper : public KeyPressEventServer, public KeyPressEventHandler {
public:
    
    KeyMapper(const KeyMap &keymap = KeyMap()) :
        keymap_(keymap)
    { }
    
    const KeyMap & keymap() const { return this->keymap_; }
    char map(KeyEvent::EventType event, int key) const { return this->keymap_.map(event, key); }

    virtual void handleKeyPress(const KeyEvent &keypress);

private:
    KeyMap keymap_;
};

} // kbd_mgr

#endif // KEY_MAPPER_H_
