/**
@file main.h
@brief header file containing functions and variables
@brief Revision 1.0
@author Orla Gilson
@date May 2015
*/

#ifndef MAIN_H
#define MAIN_H
#include "mbed.h"
#include "BMP180.h"
#include "N5110.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"

/**
@namespace lcd
@brief bus output for Nokia 5110 LCD screen
*/
N5110 lcd(p7,p8,p9,p10,p11,p13,p26);

/**
@namespace bmp180
@brief serial input for BMP180 sensor
*/
BMP180 bmp180(p28,p27);

/**
@namespace local
@brief creates local file system to write files to
*/
LocalFileSystem local("local");

/**
@namespace leds
@brief bus output for the two centre on-board leds
*/
BusOut leds(LED3,LED2);

/**
@namespace buttonleft
@brief interrupt in function for the left hand button
*/
InterruptIn buttonleft(p15);

/**
@namespace BUT2
@brief analog in function for the right hand button
*/
AnalogIn BUT2(p16);

/**
@namespace POT
@brief analog in function for the potentiometer
*/
AnalogIn POT(p20);

/**
@namespace led
@brief GPIO output for error LED
*/
DigitalOut led(p25);

/**
@namespace buzzer
@brief PWM output for piezo buzzer
*/
PwmOut buzzer(p21);

/**
@namespace LDR
@brief analog input for the light dependant resistor
*/
AnalogIn LDR(p17);

/**
Sets x axis of screen
@param nx - length of x axis
*/
int nx=84;

/**
Sets y axis of screen
@param ny - length of y axis
*/
int ny=48;

/**
Coordinates of a pixel
@param i - x value of coordinate (0-83)
@param j - y value of coordinate (0-47
*/
int i,j;

/**
* initialising a decimal variable
*/
float temperature;

/**
* initialising a decimal variable
*/
float pressure;

/** 
* initialise the counter as an integer
*/
int counter=0;

/**
* initialise the flag for the button
*/
int buttonFlag=0;

/**
* timeout used for temperature
*/
Timeout callT;

/**
* timeout used for pressure
*/
Timeout callP;

/**
* timeout used for light
*/
Timeout callL;

/**
* ticker function for temperature
*/
Ticker timerT;

/**
* ticker function for pressure
*/
Ticker timerP;

/**
* ticker function for error
*/
Ticker timerE;

/**
* ticker function for light
*/
Ticker timerL;

/**
* initialise flag for temperature
*/
int timerTFlag = 0;

/**
* initialise flag for pressure
*/
int timerPFlag = 0;

/**
* initialise flag for error
*/
int timerEFlag = 0;

/**
* initialise flag for light
*/
int timerLFlag = 0;

/** Expire Timer T; 
* timer associated with temperature; 
* sets timerTFlag to one
*/
void timerTExpired();

/** Expire Timer P; 
* timer associated with pressure; 
* sets timerPFlag to one
*/
void timerPExpired();

/** Expire Timer E; 
* timer associated with error LED; 
* sets timerEFlag to one
*/
void timerEExpired();

/** Expire Timer L;
* timer associated with error LED;
* sets timerLFlag to one
*/
void timerLEpired();

/** Press Button; 
* interrupt associated with left button; 
* sets buttonFlag to one
*/
void buttonPressed();

/** Error Function; 
* function supplies error function; 
* LED lights up every 2 seconds; 
* buzzer goes off simultaneously with the LED
*/
void error();

/** Write Data To File; 
* function to write data to the flash drive
*/
void writeDataToFile();

/** Calls Temperature; 
* function reads the temperature from the sensor; 
* prints the current temperature to the screen
*/
void callTemp();

/** Calls Pressure; 
* function reads the pressure from the sensor; 
* prints the current pressure to the screen
*/
void callPress();

/** Calls Light; 
* function reads the light from the LDR; 
* prints the current temperature to the screen
*/
void callLight();

/** Read Temperature; 
* calls callTemp after 0.1 seconds; 
* reads temperature every 60 seconds; 
* prints temperature to screen; 
* updates screen every 60 seconds with new temperature
*/
void readTemp();

/** Read Pressure; 
* calls callPress after 0.1 seconds; 
* reads pressure every 30 minutes/1800 seconds; 
* prints pressure to screen; 
* updates screen every 30 minutes with new pressure
*/
void readPress();

/** Read Light; 
* calls callLight after 0.1 seconds; 
* reads light every 60 seconds; 
* prints light to screen; 
* updates screen every 60 seconds with new light
*/
void readLight();

/** Measurement; 
* read measurement from sensor; 
* set temperature; 
* set pressure
*/
void measurement();

/** Temperature Graph; 
* read temperature every 2 seconds; 
* create temporary array; 
* read temperature values into array; 
* scale temperature array; 
* plot array
*/
void tempGraph();

/** Pressure Graph; 
* read pressure every 2 seconds; 
* create temporary array; 
* read pressure values into array; 
* scale pressure array;
* plot array
*/
void pressGraph();

/** Light Graph; 
* read light every 2 seconds; 
* create temporary array; 
* read light values into array; 
* scale light array; 
* plot array
*/
void lightGraph();

/** Menu; 
* print temperature screen if potentiometer is greater than 2/3; 
* print pressure screen if potentiometer is between 1/3 and 2/3; 
* print light screen if potentiometer is less than 1/3
*/
void menu();

/** Clear Cells; 
* cycle through pixels and set to zero
*/
void clearCells();

#endif