/**
* @file    LCDモジュール AQM0802 を制御するクラスの宣言
*/
#if !defined(LCD_AQM0802_H)

#include "mbed.h"

#define LCD_AQM0802_H


/** LCDモジュール AQM0802 制御クラス
* @brief    LCDモジュール AQM0802 をI2C通信で制御するクラス
* @date     2016/12/18
* @code
* #include "mbed.h"
* #include "LcdAqm0802.h"
*
* LcdAqm0802 lcd(p8, p9, p10);
*
* int main() {
*   wait_ms(40);    //!< LCD動作が安定するまで電源投入後40ms以上待つ必要がある
*   lcd.initialize();
*
*   lcd.puts("Hellow\r\nWorld!");
*
*   wait(1) {
*   }
* }
* @endcode
*/
class LcdAqm0802 {
public:
    static const uint8_t    SLAVE_ADDR          = 0x7c; //!< LCDのスレーブアドレス
    static const uint8_t    LINE_COUNT          = 2;    //!< 行数
    static const uint8_t    LINE_CHARA_COUNT    = 8;    //!< 行の文字数
    
    static const uint8_t    MAX_CONTRAST        = 63;   //!< コントラスト最大値
    static const uint8_t    DEFAULT_CONTRAST    = 32;   //!< コントラスト初期値
    
private:
    static const int32_t    I2C_MAX_FREQUENCY   = 400 * 1000;   //!< I2C通信の最高周波数
    static const uint8_t    LINE_HEAD_ADDR_TBL[LINE_COUNT];  //!< 行先頭アドレス

    static const uint8_t    CTRL_BYTE_RS_COMMAND   = 0x00;  //!< 制御指定用bit_RS：コマンド送信
    static const uint8_t    CTRL_BYTE_RS_DATA      = 0x40;  //!< 制御指定用bit_RS：データ送信
    static const uint8_t    CTRL_BYTE_CO_SINGLE    = 0x80;  //!< 制御指定用bit_Co：単発
    static const uint8_t    CTRL_BYTE_CO_CHAIN     = 0x00;  //!< 制御指定用bit_Co：連続
    
    static const uint8_t    ADDR_BYTE_WRITE             = SLAVE_ADDR;           //!< アドレス指定用バイトデータ：書き込み
    static const uint8_t    ADDR_BYTE_READ              = SLAVE_ADDR | 0x01;    //!< アドレス指定用バイトデータ：読み込み
    static const uint8_t    CTRL_BYTE_SINGLE_COMMAND    = CTRL_BYTE_RS_COMMAND | CTRL_BYTE_CO_SINGLE;   //!< 制御指定用バイトデータ：単発コマンド
    static const uint8_t    CTRL_BYTE_CHAIN_COMMAND     = CTRL_BYTE_RS_COMMAND | CTRL_BYTE_CO_CHAIN;    //!< 制御指定用バイトデータ：連続コマンド
    static const uint8_t    CTRL_BYTE_SINGLE_DATA       = CTRL_BYTE_RS_DATA | CTRL_BYTE_CO_SINGLE;      //!< 制御指定用バイトデータ：単発データ
    static const uint8_t    CTRL_BYTE_CHAIN_DATA        = CTRL_BYTE_RS_DATA | CTRL_BYTE_CO_CHAIN;       //!< 制御指定用バイトデータ：連続データ
    
    static const uint8_t    FUNC_SET_BYTE_EX    = 0x39; //!< ファンクションセット用バイト：拡張設定
    static const uint8_t    FUNC_SET_BYTE_NRM   = 0x38; //!< ファンクションセット用バイト：通常設定

private:
    PinName     m_pinReset;     //!< リセットピン
    PinName     m_pinSDA;       //!< SDAピン
    PinName     m_pinSCL;       //!< SCLピン
    DigitalOut  m_resetOut;    //!< リセット制御
    I2C         m_i2c;         //!< I2C制御
    
    bool            m_displayVisible;    //!< ディスプレイ表示フラグ
    bool            m_cursorVisible;     //!< カーソル表示フラグ
    bool            m_blinkEnable;       //!< カーソル点滅フラグ
    
    uint8_t m_contrast; //!< コントラスト
    uint8_t m_currentLineNo;  //!< 現在位置：行
    uint8_t m_currentCharaNo; //!< 現在位置：文字
public:
    /** コンストラクタ
    * @brief    クラス内のI2Cインスタンスを使用する場合のコンストラクタ
    * @param    pinReset    リセットピン指定
    * @param    pinSDA      SDAピン指定
    * @param    pinSCL      SCLピン指定
    */
    LcdAqm0802(PinName pinReset, PinName pinSDA, PinName pinSCL);
    /** コンストラクタ
    * @brief    クラス外のI2Cインスタンスを使用する場合のコンストラクタ
    * @param    pinReset    リセットピン指定
    * @param    i2c         I2Cインスタンス
    */
    LcdAqm0802(PinName pinReset, I2C& i2c);
    /** デストラクタ
    */
    ~LcdAqm0802();
    
    
    /** 初期化
    * @brief    LCDの表示に必要な各種初期化を行います。
    *           これを行わないと表示されません。
    *           ※LCDに電源投入後（もしくはリセット後）、必ず40ms以上経過後に呼び出してください。
    */
    void initialize();
    /** リセット
    * @brief    LCDを初期化前の状態に戻します。
    */
    void reset();
    
    
    /** 1文字描画
    *   @brief  1文字描画します。
    *           カーソル位置は一つ進みます。
    *           行端まで来た場合は自動で復帰・改行します。
    *   @param  c   文字コード
    */
    void putc(char c);
    /** 文字列描画
    *   @brief  文字列の終端まで描画します。
    *           カーソル位置は文字数分進みます。
    *           行端まで来た場合は自動で復帰・改行します。
    *   @param  pStr    文字列
    */
    void puts(const char* pStr);
    /** フォーマット指定描画
    *   @brief  フォーマットと可変長引数から生成した文字列を描画します。
    *           文字列生成後、puts()を内部で呼び出しています。
    *   @param  pFormat フォーマット
    *   @param  ...     可変長引数
    */
    void printf(const char* pFormat, ...);    
    
    
    /** カーソル位置変更
    *   @brief  指定位置までカーソルを移動させます。
    *   @param  lineNo  行番号（0～LINE_COUNT）
    *   @param  charaNo 文字番号（0～LINE_CHARA_COUNT）
    */
    void locate(uint8_t lineNo, uint8_t charaNo);
    /** カーソル位置取得：行番号
    *   @return カーソルのある行番号
    */
    uint8_t getCurrentLineNo() const;
    /** カーソル位置取得：文字番号
    *   @return カーソルのある文字番号
    */
    uint8_t getCurrentCharaNo() const;


    /** 画面クリア
    *   @brief  画面全域をクリアし、カーソル位置を先頭に移動させます。
    */
    void clearDisplay();
    /** 行クリア
    *   @brief  指定行をクリアし、カーソルを指定行の先頭に移動させます。
    *   @param  lineNo  行番号
    */
    void clearLine(uint8_t lineNo);
    /** 範囲クリア
    *   @brief  指定範囲をクリアし、カーソルを指定範囲の開始位置に移動させます。
    *   @param  startLineNo     開始行番号
    *   @param  startCharaNo    開始文字番号
    *   @param  endLineNo;      終了行番号
    *   @param  endCharaNo      終了文字番号
    */
    void clearRange(uint8_t startLineNo, uint8_t startCharaNo, uint8_t endLineNo, uint8_t endCharaNo);


    /** 画面の表示設定
    *   @param  visible    表示フラグ
    */
    void setDisplayVisible(bool visible);
    /** 画面の表示設定取得
    *   @return 表示フラグ
    */
    bool getDisplayVisible() const;
    /** カーソルの表示設定
    *   @param  visible 表示フラグ
    */
    void setCursorVisible(bool visible);
    /** カーソルの表示設定取得
    *   @return 表示フラグ
    */
    bool getCursorVisible() const;
    /** カーソルの点滅設定
    *   @param  enable  点滅フラグ
    */
    void setBlinkEnable(bool enable);
    /** カーソルの点滅設定取得
    *   @return 点滅フラグ
    */
    bool getBlinkEnable() const;

    
    /** コントラスト設定
    * @param    val コントラスト値（0～MAX_CONTRAST）
    */
    void setContrast(uint8_t val);
    /** コントラスト取得
    * @return   コントラスト値
    */
    uint8_t getContrast() const;
private:
    /** LCDコマンド単発書き込み
    *   @brief  LCDのコマンドを一つ書き込み（実行し）ます。
    *   @param  commandByte コマンド指定バイトデータ
    */
    void writeCommand(uint8_t commandByte);
    /** LCDコマンド連続書き込み
    *   @brief  LCDのコマンドを複数書き込み（実行し）ます。
    *   @param  commandBytes コマンド指定バイトデータ配列
    *   @param  commandCount コマンド指定バイトデータ数
    */
    void writeCommand(uint8_t commandBytes[], uint8_t commandCount);
    /** LCDデータ単発書き込み
    *   @brief  LCDのデータを一つ書き込み（実行し）ます。
    *   @param  dataByte データ指定バイトデータ
    */
    void writeData(uint8_t dataByte);
    /** LCDデータ連続書き込み
    *   @brief  LCDのデータを複数書き込み（実行し）ます。
    *   @param  dataBytes データ指定バイトデータ配列
    *   @param  dataCount データ指定バイトデータ数
    */
    void writeData(uint8_t dataBytes[], uint8_t dataCount);


    /** コントラスト設定用バイトデータ生成（上位ビット）
    * @return   コントラスト設定用バイトデータ（上位ビット）
    */
    uint8_t calcContrastSettingByteUpper() const;
    /** コントラスト設定用バイトデータ生成（下位ビット）
    * @return   コントラスト設定用バイトデータ（下位ビット）
    */
    uint8_t calcContrastSettingByteLowwer() const;


    /** ディスプレイ制御コマンド指定バイトデータの生成
    *   @param  displayVisible  ディスプレイ表示設定
    *   @param  cursorVisible   カーソル表示設定
    *   @param  blinkEnable     カーソル点滅設定
    *   @return コマンド指定バイトデータ
    */
    uint8_t calcDisplayControlByteData(bool displayVisible, bool cursorVisible, bool blinkEnable) const;
    uint8_t calcDisplayControlByteData() const {
        return calcDisplayControlByteData(m_displayVisible, m_cursorVisible, m_blinkEnable);
    }

    
    /** カーソル位置移動管理
    *   @brief  カーソル位置を一つ進め、改行が発生するかを調べる
    *   @return 改行が発生する場合はtrue
    */
    bool incCursorPos();

    
    /** カーソル位置のアドレス計算
    *   @param  lineNo  行番号
    *   @param  charaNo 文字番号
    *   @return アドレス
    */
    uint8_t calcCursorAddr(uint8_t lineNo, uint8_t charaNo);
};

#endif  //!defined(LCD_AQM0802_H)
