#if !defined(LCD_ACM1602NI_H)

#include "mbed.h"

#define LCD_ACM1602NI_H

/** LCDモジュール ACM1602NI を制御するクラス
*   @brief  ACM1602NIをI2C通信で制御します。
*   @date   2016/12/14
*   @author H.Nagahama
*   @code
*   #include "mbed.h"
*   #include "LcdAcm1602ni.h"
*   
*   LcdAcm1602ni lcd(p9, p10);
*   
*   int main() {
*       wait_ms(15);    //!< LCDの安定まで15ms以上待つ必要がある
*       lcd.initialize();
*       
*       lcd.puts("Hellow World!");
*       
*       while(1) {
*       }
*   }
*   @endcode
*/
class LcdAcm1602ni
{
public:
    static const uint8_t    SLAVE_ADDR          = 0xa0;     //!< スレーブアドレス（下位ビットに0を追加済み）
    static const uint8_t    LINE_COUNT          = 2;        //!< 行数
    static const uint8_t    LINE_CHARA_COUNT    = 16;       //!< 行の文字数
    
private:
    static const int32_t    I2C_MAX_FREQUENCY   = 70 * 1000;   //!< I2C通信の最大周波数
    static const uint8_t    LINE_HEAD_ADDR_TBL[LINE_COUNT];     //!< 行の先頭アドレス
    
    static const uint8_t    ADDR_BYTE_WRITE     = SLAVE_ADDR;           //!< アドレス指定用バイトデータ：書き込み用
    static const uint8_t    ADDR_BYTE_READ      = SLAVE_ADDR & 0x01;    //!< アドレス指定用バイトデータ：読み込み用
    static const uint8_t    CTRL_BYTE_COMMAND   = 0x00;                 //!< 制御指定用バイトデータ：コマンド用
    static const uint8_t    CTRL_BYTE_DATA      = 0x80;                 //!< 制御指定用バイトデータ：データ用

private:
    PinName m_pinSDA;   //!< SDAピン名
    PinName m_pinSCL;   //!< SCLピン名
    I2C     m_i2c;      //!< I2C通信制御
    bool    m_isMyI2c;  //!< 自前のI2Cクラス使用フラグ
    
    uint8_t m_currentLineNo;    //!< 現在位置：行番号
    uint8_t m_currentCharaNo;   //!< 現在位置：文字番号
    
    bool    m_displayVisible;   //!< ディスプレイ表示フラグ
    bool    m_cursorVisible;    //!< カーソル表示フラグ
    bool    m_blinkEnable;      //!< カーソル点滅フラグ
    
public:
    /** コンストラクタ
    *   @brief  自前のI2Cインスタンスを作成する場合のコンストラクタ
    *           I2Cの動作周波数はI2C_MAX_FREQUENCYを設定します。
    *   @param  pinSDA  SDAピン指定
    *   @param  pinSCL  SCLピン指定
    */
    LcdAcm1602ni(PinName pinSDA, PinName pinSCL);
    /** コンストラクタ
    *   @brief  外部のI2Cインスタンスを共有する場合のコンストラクタ
    *           本クラスがI2Cの設定を変更することはありません。
    *   @param  i2c I2Cインスタンス
    */
    LcdAcm1602ni(I2C& i2c);
    /** デストラクタ
    */
    ~LcdAcm1602ni();
    
    
    /** 初期化
    *   @brief  LCDを初期化し、表示可能な状態にします。
    *           電源投入後、LCDが安定するまで15ms以上待ってから実行してください。
    */
    void initialize();
    
    
    /** 1文字描画
    *   @brief  1文字描画します。
    *           カーソル位置は一つ進みます。
    *           行端まで来た場合は自動で復帰・改行します。
    *   @param  c   文字コード
    */
    void putc(char c);
    /** 文字列描画
    *   @brief  文字列の終端まで描画します。
    *           カーソル位置は文字数分進みます。
    *           行端まで来た場合は自動で復帰・改行します。
    *   @param  pStr    文字列
    */
    void puts(const char* pStr);
    /** フォーマット指定描画
    *   @brief  フォーマットと可変長引数から生成した文字列を描画します。
    *           文字列生成後、puts()を内部で呼び出しています。
    *   @param  pFormat フォーマット
    *   @param  ...     可変長引数
    */
    void printf(const char* pFormat, ...);


    /** カーソル位置変更
    *   @brief  指定位置までカーソルを移動させます。
    *   @param  lineNo  行番号（0～LINE_COUNT）
    *   @param  charaNo 文字番号（0～LINE_CHARA_COUNT）
    */
    void locate(uint8_t lineNo, uint8_t charaNo);
    /** カーソル位置取得：行番号
    *   @return カーソルのある行番号
    */
    uint8_t getCurrentLineNo() const;
    /** カーソル位置取得：文字番号
    *   @return カーソルのある文字番号
    */
    uint8_t getCurrentCharaNo() const;
    
    
    /** 画面クリア
    *   @brief  画面全域をクリアし、カーソル位置を先頭に移動させます。
    */
    void clearDisplay();
    /** 行クリア
    *   @brief  指定行をクリアし、カーソルを指定行の先頭に移動させます。
    *   @param  lineNo  行番号
    */
    void clearLine(uint8_t lineNo);
    /** 範囲クリア
    *   @brief  指定範囲をクリアし、カーソルを指定範囲の開始位置に移動させます。
    *   @param  startLineNo     開始行番号
    *   @param  startCharaNo    開始文字番号
    *   @param  endLineNo;      終了行番号
    *   @param  endCharaNo      終了文字番号
    */
    void clearRange(uint8_t startLineNo, uint8_t startCharaNo, uint8_t endLineNo, uint8_t endCharaNo);
    
    
    /** 画面の表示設定
    *   @param  visible    表示フラグ
    */
    void setDisplayVisible(bool visible);
    /** 画面の表示設定取得
    *   @return 表示フラグ
    */
    bool getDisplayVisible() const;
    /** カーソルの表示設定
    *   @param  visible 表示フラグ
    */
    void setCursorVisible(bool visible);
    /** カーソルの表示設定取得
    *   @return 表示フラグ
    */
    bool getCursorVisible() const;
    /** カーソルの点滅設定
    *   @param  enable  点滅フラグ
    */
    void setBlinkEnable(bool enable);
    /** カーソルの点滅設定取得
    *   @return 点滅フラグ
    */
    bool getBlinkEnable() const;
    
private:
    /** LCDコマンド書き込み
    *   @brief  LCDのコマンドを一つ書き込み（実行し）ます。
    *   @param  commandByte コマンド指定バイトデータ
    */
    void writeCommand(uint8_t commandByte);
    /** LCDデータ書き込み
    *   @brief  LCDへ文字データを一つ書き込みます。
    *   @param  dataByte    データ指定バイト
    */
    void writeData(uint8_t  dataByte);


    /** ディスプレイ制御コマンド指定バイトデータの生成
    *   @param  displayVisible  ディスプレイ表示設定
    *   @param  cursorVisible   カーソル表示設定
    *   @param  blinkEnable     カーソル点滅設定
    *   @return コマンド指定バイトデータ
    */
    uint8_t calcDisplayControlByteData(bool displayVisible, bool cursorVisible, bool blinkEnable) const;
    uint8_t calcDisplayControlByteData() const {
        return calcDisplayControlByteData(m_displayVisible, m_cursorVisible, m_blinkEnable);
    }
    
    
    /** カーソル位置移動管理
    *   @brief  カーソル位置を一つ進め、改行が発生するかを調べる
    *   @return 改行が発生する場合はtrue
    */
    bool incCursorPos();
    
    /** カーソル位置のアドレス計算
    *   @param  lineNo  行番号
    *   @param  charaNo 文字番号
    *   @return アドレス
    */
    uint8_t calcCursorAddr(uint8_t lineNo, uint8_t charaNo);
};

#endif  //!defined(LCD_ACM1602NI_H)
