/** PID.cpp
 * UWB Quadcopter project
 *
 * Implementation of the Proportional Integral Derivative
 * control system for the quadcopter.
 * http://nicisdigital.wordpress.com/2011/06/27/proportional-integral-derivative-pid-controller/
 *
 * @author UWB Quadcopter group
 */

#include "PID.h"
//
//#include "mbed.h"
//Serial pc(USBTX, USBRX);

PID::PID(const float proportionalGain,
    const float integralGain,
    const float derivativeGain,
    const float windupGainGuard ) :
        proportionalGain(proportionalGain),
        integralGain(integralGain),
        derivativeGain(derivativeGain),
        windupGainGuard(windupGainGuard/integralGain)
{
//    pc.baud(38400);
    integralError = 0.0f;
    previousError = 0.0f;
    currentError = 0.0f;
    proportionalControl = integralControl = derivativeControl = 0;
}

float PID::correct(const float currentError,const int dt)
{
    // Sum of the errors the controller has incurred over time
    integralError += currentError * dt / 1000.0f;

    // Cap the sum of errors to prevent excessive correction
    if (integralError < -windupGainGuard) {
        integralError = -windupGainGuard;
    } else if (integralError > windupGainGuard) {
        integralError = windupGainGuard;
    }
    
    // Calculate the correction, based on the tuning values
    proportionalControl = proportionalGain * currentError;
    integralControl = integralError * integralGain;
    derivativeControl = derivativeGain * (currentError - previousError) / dt * 1000.0f;

    // Save current error for next call to correction
    previousError = currentError;
    
    controlTotal = proportionalControl + integralControl + derivativeControl;
    
    if (controlTotal > CONTROL_LIMIT)
        return CONTROL_LIMIT;
    else if (controlTotal < -CONTROL_LIMIT)
        return -CONTROL_LIMIT;
        
    return controlTotal;

}

void PID::updateP(const float pGain)
{
    proportionalGain = pGain;
}

void PID::updateD(const float dGain)
{
    derivativeGain = dGain;
}
