/* vi: set sw=4 ts=4: */
/*
 * crond -d[#] -c <crondir> -f -b
 *
 * run as root, but NOT setuid root
 *
 * Copyright 1994 Matthew Dillon (dillon@apollo.west.oic.com)
 * (version 2.3.2)
 * Vladimir Oleynik <dzo@simtreas.ru> (C) 2002
 *
 * Licensed under GPLv2 or later, see file LICENSE in this source tree.
 */

#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <cstring>
#include "crond.h"
#include "crond_config.h"

#ifdef __unix__                    /* __unix__ is usually defined by compilers targeting Unix systems */
    #include <unistd.h>
    #include <stdlib.h>
    #define MSEC_SLEEP(MSEC)  msleep(MSEC)
#elif defined(_WIN32) || defined(WIN32)     /* _Win32 is usually defined by compilers targeting 32 or   64 bit Windows systems */
    #include <windows.h>
    #define MSEC_SLEEP(MSEC)  Sleep(MSEC)
#else
	#include "mbed.h" 
	#define MSEC_SLEEP(MSEC)  wait_ms(MSEC)
#endif

#define TO_MSEC(sec)  ((1000/60) * (sec))
#define ARRAY_SIZE(x) ((unsigned)(sizeof(x) / sizeof((x)[0])))
#define PROCESS_FINISHED_JOB(crontab_name, line)  ((line)->cl_pid = 0)
#define G   (*this)

static const char DowAry[] =
		"sun""mon""tue""wed""thu""fri""sat";
		/* "Sun""Mon""Tue""Wed""Thu""Fri""Sat" */
	

static const char MonAry[] =
	"jan""feb""mar""apr""may""jun""jul""aug""sep""oct""nov""dec"
	/* "Jan""Feb""Mar""Apr""May""Jun""Jul""Aug""Sep""Oct""Nov""Dec" */
;

static void crond_sleep(int msec, void* tag)
{
	 MSEC_SLEEP(msec);
}

void Crond::crondlog(const char *ctl, ...)
{
	char level = ctl[0];
    va_list arglist;
    va_start(arglist, ctl);
    CRON_VPRINTF(level, &ctl[1], arglist);
    va_end(arglist);
    return;
}

void Crond::parse_field(char *user, char *ary, int modvalue, int off,
				const char *names, char *ptr)
/* 'names' is a pointer to a set of 3-char abbreviations */
{
	char *base = ptr;
	int n1 = -1;
	int n2 = -1;

	// this can't happen due to config_read()
	/*if (base == NULL)
		return;*/

	while (1) {
		int skip = 0;

		/* Handle numeric digit or symbol or '*' */
		if (*ptr == '*') {
			n1 = 0;  /* everything will be filled */
			n2 = modvalue - 1;
			skip = 1;
			++ptr;
		} else if (isdigit(*ptr)) {
			char *endp;
			if (n1 < 0) {
				n1 = strtol(ptr, &endp, 10) + off;
			} else {
				n2 = strtol(ptr, &endp, 10) + off;
			}
			ptr = endp; /* gcc likes temp var for &endp */
			skip = 1;
		} else if (names) {
			int i;

			for (i = 0; names[i]; i += 3) {
				/* was using strncmp before... */
				if (strncasecmp(ptr, &names[i], 3) == 0) {
					ptr += 3;
					if (n1 < 0) {
						n1 = i / 3;
					} else {
						n2 = i / 3;
					}
					skip = 1;
					break;
				}
			}
		}

		/* handle optional range '-' */
		if (skip == 0) {
			goto err;
		}
		if (*ptr == '-' && n2 < 0) {
			++ptr;
			continue;
		}

		/*
		 * collapse single-value ranges, handle skipmark, and fill
		 * in the character array appropriately.
		 */
		if (n2 < 0) {
			n2 = n1;
		}
		if (*ptr == '/') {
			char *endp;
			skip = strtol(ptr + 1, &endp, 10);
			ptr = endp; /* gcc likes temp var for &endp */
		}

		/*
		 * fill array, using a failsafe is the easiest way to prevent
		 * an endless loop
		 */
		{
			int s0 = 1;
			int failsafe = 1024;

			--n1;
			do {
				n1 = (n1 + 1) % modvalue;

				if (--s0 == 0) {
					ary[n1 % modvalue] = 1;
					s0 = skip;
				}
				if (--failsafe == 0) {
					goto err;
				}
			} while (n1 != n2);
		}
		if (*ptr != ',') {
			break;
		}
		++ptr;
		n1 = -1;
		n2 = -1;
	}

	if (*ptr) {
 err:
		crondlog(WARN9 "user %s: parse error at %s"CR, user, base);
		return;
	}

	if (DebugOpt && (G.log_level <= 5)) { /* like LVL5 */
		/* can't use crondlog, it inserts '\n' */
		int i;
		for (i = 0; i < modvalue; ++i)
			crondlog("ary[%d] = %d"CR, i, (unsigned char)ary[i]);
	}
}

void Crond::fix_day_dow(CronLine *line)
{
	unsigned i;
	int weekUsed = 0;
	int daysUsed = 0;

	for (i = 0; i < ARRAY_SIZE(line->cl_Dow); ++i) {
		if (line->cl_Dow[i] == 0) {
			weekUsed = 1;
			break;
		}
	}
	for (i = 0; i < ARRAY_SIZE(line->cl_Days); ++i) {
		if (line->cl_Days[i] == 0) {
			daysUsed = 1;
			break;
		}
	}
	if (weekUsed != daysUsed) {
		if (weekUsed)
			std::memset(line->cl_Days, 0, sizeof(line->cl_Days));
		else /* daysUsed */
			std::memset(line->cl_Dow, 0, sizeof(line->cl_Dow));
	}
}

void Crond::start_one_job(CronLine *line)
{
	line->cl_pid = line->cl_exec_func(line->cl_func_arg);
}

/*
 * Determine which jobs need to be run.  Under normal conditions, the
 * period is about a minute (one scan).  Worst case it will be one
 * hour (60 scans).
 */
void Crond::flag_starting_jobs(time_t t1, time_t t2)
{
	time_t t;

	/* Find jobs > t1 and <= t2 */

	for (t = t1 - t1 % 60; t <= t2; t += 1) {
		struct tm *ptm;
		CronFile *file;
		CronLine *line;

		if (t <= t1)
			continue;

		ptm = localtime(&t);
		for (file = G.cron_files; file; file = file->cf_next) {
			if (DebugOpt && (G.log_level <= 5))
				crondlog(LVL5 "file %s:"CR, file->cf_crontabset_name);
			if (file->cf_deleted)
				continue;
			for (line = file->cf_lines; line; line = line->cl_next) {
				if (line->cl_Secs[ptm->tm_sec]
				 && line->cl_Mins[ptm->tm_min]
				 && line->cl_Hrs[ptm->tm_hour]
				 && (line->cl_Days[ptm->tm_mday] || line->cl_Dow[ptm->tm_wday])
				 && line->cl_Mons[ptm->tm_mon]
				) {
					if (line->cl_pid == 0) {
						line->cl_pid = -1;
						file->cf_wants_starting = 1;
					}
				}
			}
		}
	}
}

void Crond::start_jobs(void)
{
	CronFile *file;
	CronLine *line;

	for (file = G.cron_files; file; file = file->cf_next) {
		if (!file->cf_wants_starting)
			continue;

		file->cf_wants_starting = 0;
		for (line = file->cf_lines; line; line = line->cl_next) {
			int pid;
			if (line->cl_pid >= 0)
				continue;

			start_one_job(line);
			pid = line->cl_pid;
			crondlog(LVL8 "USER %s pid %3d exec %p arg %p"CR,
				file->cf_crontabset_name, (int)pid, line->cl_exec_func, line->cl_func_arg);
			if (pid < 0) {
				file->cf_wants_starting = 1;
			}
			if (pid > 0) {
				file->cf_has_running = 1;
			}
		}
	}
}

/*
 * Check for job completion, return number of jobs still running after
 * all done.
 */
int Crond::check_completions(void)
{
	CronFile *file;
	CronLine *line;
	int num_still_running = 0;

	for (file = G.cron_files; file; file = file->cf_next) {
		if (!file->cf_has_running)
			continue;

		file->cf_has_running = 0;
		for (line = file->cf_lines; line; line = line->cl_next) {
			int r;

			if (line->cl_pid <= 0)
				continue;

			r = line->cl_wait_func(line->cl_pid, line->cl_func_arg);
			if (r < 0 || r == line->cl_pid) {
				PROCESS_FINISHED_JOB(file->cf_crontabset_name, line);
				if (line->cl_pid == 0) {
					/* sendmail was not started for it */
					continue;
				}
				/* else: sendmail was started, job is still running, fall thru */
			}
			/* else: r == 0: "process is still running" */
			file->cf_has_running = 1;
		}
//FIXME: if !file->cf_has_running && file->deleted: delete it!
//otherwise deleted entries will stay forever, right?
		num_still_running += file->cf_has_running;
	}
	return num_still_running;
}

Crond::~Crond(void)
{
	int len;
	len = G.length();
	
	G.stop();
	while(len > 0)
	{
		G.delete_crontab(G.get_crontab_name(0));
		len--;
	}
}

////////////////// public methods /////////////////////////

Crond::Crond(void (*my_sleep)(int, void*), void *tag)
{
	G.status = STOPPING;
	G.log_level = 8;
	G.cron_files = NULL;
	G.sleep_tag = tag;
	if (my_sleep != NULL) {
		G.sleep = my_sleep;
	} else {
		G.sleep = crond_sleep;
	}
}

int Crond::add_crontab(const char* crontabset_name, const CronTab crontab[], int length)
{
	delete_crontab(crontabset_name);
	
	CronFile *file;
	CronLine **pline;
    
	file = (CronFile *)calloc(1,sizeof(CronFile));
	if (file == NULL) {
		crondlog(ERR20"memory alloc error %s"CR, crontabset_name);
		return (-1);
	}

    if ((file->cf_crontabset_name = (char*)malloc(sizeof(char) * (std::strlen(crontabset_name) + 1))) != NULL) {
        std::strcpy(file->cf_crontabset_name, crontabset_name);
    } else {
        crondlog(ERR20"memory alloc error %s"CR, crontabset_name);
        free(file);
        return (-1);
    }
    
    if (G.log_level <= 8) {
    	crondlog(LVL8"add crontab %s"CR, crontabset_name);
    }
    
	pline = &file->cf_lines;

	for(int i=0; i < length; i++) {
		CronLine *line;
		line = (CronLine *)calloc(1, sizeof(*line));
		if (line != NULL)
		{
			*pline = line;
			/* parse date ranges */
			parse_field(file->cf_crontabset_name, line->cl_Secs, 60, 0, NULL, crontab[i].seconds);
			parse_field(file->cf_crontabset_name, line->cl_Mins, 60, 0, NULL, crontab[i].minits);
			parse_field(file->cf_crontabset_name, line->cl_Hrs, 24, 0, NULL, crontab[i].hours);
			parse_field(file->cf_crontabset_name, line->cl_Days, 32, 0, NULL, crontab[i].days);
			parse_field(file->cf_crontabset_name, line->cl_Mons, 12, -1, MonAry, crontab[i].mounth);
			parse_field(file->cf_crontabset_name, line->cl_Dow, 7, 0, DowAry, crontab[i].day_of_week);
			/*
			 * fix days and dow - if one is not "*" and the other
			 * is "*", the other is set to 0, and vise-versa
			 */
			fix_day_dow(line);
			/* copy function */
			line->cl_exec_func = crontab[i].exec_func;
			line->cl_func_arg = crontab[i].func_arg;
			line->cl_wait_func = crontab[i].wait_func;
			
			pline = &line->cl_next;
		} else {
			crondlog(ERR20"memory alloc error %s must free!"CR, crontabset_name);
			break;
		}
	}
	*pline = NULL;

	file->cf_next = G.cron_files;
	G.cron_files = file;
	
	return 0;
}

/*
 * delete_crontab() - delete user database
 *
 * Note: multiple entries for same user may exist if we were unable to
 * completely delete a database due to running processes.
 */
//FIXME: we will start a new job even if the old job is running
//if crontab was reloaded: crond thinks that "new" job is different from "old"
//even if they are in fact completely the same. Example
//Crontab was:
// 0-59 * * * * job1
// 0-59 * * * * long_running_job2
//User edits crontab to:
// 0-59 * * * * job1_updated
// 0-59 * * * * long_running_job2
//Bug: crond can now start another long_running_job2 even if old one
//is still running.
//OTOH most other versions of cron do not wait for job termination anyway,
//they end up with multiple copies of jobs if they don't terminate soon enough.
void Crond::delete_crontab(const char *crontabset_name)
{
	CronFile **pfile = &G.cron_files;
	CronFile *file;

	while ((file = *pfile) != NULL) {
		if (std::strcmp(crontabset_name, file->cf_crontabset_name) == 0) {
		    if (G.log_level <= 8) {
		    	crondlog(LVL8"delete crontab %s"CR, crontabset_name);
		    }
			
			CronLine **pline = &file->cf_lines;
			CronLine *line;

			file->cf_has_running = 0;
			file->cf_deleted = 1;

			while ((line = *pline) != NULL) {
				if (line->cl_pid > 0) {
					file->cf_has_running = 1;
					pline = &line->cl_next;
				} else {
					*pline = line->cl_next;
					free(line);
				}
			}
			if (file->cf_has_running == 0) {
				*pfile = file->cf_next;
				free(file->cf_crontabset_name);
				free(file);
				continue;
			}
		}
		pfile = &file->cf_next;
	}
}

int Crond::length(void)
{
	CronFile *file;
	int count = 0;
	for (file = G.cron_files; file; file = file->cf_next) {
		count++;
	}
	return count;
}

const char* Crond::get_crontab_name(int idx)
{
	const char* name = NULL;
	CronFile *file;
	
	for (file = G.cron_files; (file != NULL) && (idx >= 0); file = file->cf_next) {
		name = file->cf_crontabset_name;
		idx--;
	}
	return name;
}

int Crond::start(void)
{
	time_t t2;
	int sleep_time;

	G.status = RUNNING;

	crondlog(LVL8 "crond started, log level %d"CR, G.log_level);

	/* Main loop */
	t2 = time(NULL);
	sleep_time = 60;
	while (G.status == RUNNING) {
		time_t t1;
		long dt;

		t1 = t2;

		/* Synchronize to 1 minute, minimum 1 second */
		G.sleep(TO_MSEC(sleep_time - (time(NULL) % sleep_time) + 1), G.sleep_tag);

		t2 = time(NULL);
		dt = (long)t2 - (long)t1;

		/*
		 * The file 'cron.update' is checked to determine new cron
		 * jobs.  The directory is rescanned once an hour to deal
		 * with any screwups.
		 *
		 * Check for time jump.  Disparities over an hour either way
		 * result in resynchronization.  A negative disparity
		 * less than an hour causes us to effectively sleep until we
		 * match the original time (i.e. no re-execution of jobs that
		 * have just been run).  A positive disparity less than
		 * an hour causes intermediate jobs to be run, but only once
		 * in the worst case.
		 *
		 * When running jobs, the inequality used is greater but not
		 * equal to t1, and less then or equal to t2.
		 */
		if (DebugOpt && (G.log_level <= 5))
			crondlog(LVL5 "wakeup dt=%ld"CR, dt);
		if (dt < -60 * 60 || dt > 60 * 60) {
			crondlog(WARN9 "time disparity of %ld minutes detected"CR, dt / 60);
			/* and we do not run any jobs in this case */
		} else if (dt > 0) {
			/* Usual case: time advances forward, as expected */
			flag_starting_jobs(t1, t2);
			start_jobs();
			if (check_completions() > 0) {
				/* some jobs are still running */
				sleep_time = 10;
			} else {
				sleep_time = 60;
			}
		}
		/* else: time jumped back, do not run any jobs */
	} /* while */

	return 0; /* not reached */
}

void Crond::stop(void)
{
	G.status = STOPPING;
}
