#ifndef DS2_H
#define DS2_H

#include "mbed.h"

#include "Bus.h"

extern Serial dbg;

#define DS2_MTU (127)

static const char DS2_IDENTIFY[] = {0x00};

/*
a ds2 packet looks like this:
{0xAA, 0xAA, 0xLL, 0xDD, 0xCC}
or this:
{0xAA, 0xLL, 0xDD, 0xCC}
AA = module address
LL = packet length
DD = data
CC = checksum
there can be either one or two AA module address bytes
there can be 1 or more DD data bytes

here is an example query packet:
{0x12, 0x04, 0x00, 0x16} (query DME for identification)
the address is a one-byte address of 0x12 (DME)
the packet length is 0x04 bytes
the one byte data payload is 0x00 (identify module)
the xor checksum of the entire packet is 0x16

the reply would look like this:
{0x12, 0x10, ... 0x??} (DME responding...)
the one byte address is 0x12 (DME)
the packet length will be 0x10 bytes
the xor checksum 0x?? would depend on the missing data "..."

the same query packet would look like this with two address bytes:
{0x12, 0x00, 0x05, 0x00, 0x17}

it seems that some modules want one address byte and some want two
either that or it's an extra length byte we're dealing with
probably it's a length byte, but I'll worry about it when I find out
*/
class DS2Packet
{
    unsigned short* address;
    char* length;
    char* data;
    char* checksum;
    
    char* rawdata;
    bool has8BitAddr;

    void updateChecksum();
    
public:
    DS2Packet(int maxlength);
    DS2Packet(int address, const char* data, int length);
    ~DS2Packet();
    
    void has8BitAddress(bool b);
    
    bool has8BitAddress()
    {
        return has8BitAddr;
    }
    
    int getAddress()
    {
        if(has8BitAddr)
            return *address >> 8 & 0xff;
        return *address;
    }
    
    int getLength()
    {
        return *length;
    }
    
    char* getData()
    {
        return data;
    }

    int getChecksum()
    {
        char* check = rawdata + *length - 1;
        return *check;
    }
    
    char* getRawData()
    {
        return rawdata;
    }
};

class DS2
{
    Bus *k;
    Bus *l;
    
public:
    DS2(Bus* KLine, Bus* LLine);
    ~DS2();
    int sendPacket(DS2Packet* packet, Bus* bus);
    DS2Packet* getPacket(Bus* bus);
    DS2Packet* getPacket8(Bus* bus);
    bool test();
    void testModule(int address);
    void snoop();

};

#endif //DS2_H