#ifndef HT16K33_H
#define HT16K33_H
 
#include "mbed.h"

#define HT16K33_ADDRESS         0x70<<1  // shift to the left one for mbed
#define HT16K33_ON              0x21  // Commands
#define HT16K33_STANDBY         0x20
#define HT16K33_DISPLAYON       0x81
#define HT16K33_DISPLAYOFF      0x80
#define HT16K33_BLINKON         0x85 // Blink is off (00), 2 Hz (01), 1 Hz (10), or 0.5 Hz (11) for bits (21) 
#define HT16K33_BLINKOFF        0x81
#define HT16K33_DIM             0xE0 | 0x08  // Set brihtness from 1/16 (0x00) to 16/16 (0xFF)

// Arrangement for display 1 (4 digit bubble display)
// 
//               a = A0
//             _________
//            |         |
//   f = A2   |  g = A4 | b = A1
//            |_________|
//            |         |
//   e = A5   |         | c = A6
//            |_________|
//               d = A3        DP = A7


static const char numberTable[] =
{
  0x6F, // 0 = 0
  0x42, // 1 = 1, etc
  0x3B, // 2
  0x5B, // 3
  0x56, // 4
  0x5D, // 5
  0x7D, // 6
  0x43, // 7
  0x7F, // 8
  0x57, // 9
  0x80, // decimal point
  0x00, // blank
  0x10, // minus sign
};

#define display1 1
#define display2 2
#define display3 3
#define display4 4

class HT16K33 {
 
    protected:
 
    public: 
 //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//++ Useful Functions++
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void writeInteger(uint8_t dsply, int data)
{
  char string[10] = "";                             // define character array to hold the digits
  itoa(data, string);                               // get ascii character string representation of the integer to be displayed
  uint8_t length = strlen(string);                  // get the length of the string; number of digits in integer
  uint8_t blanks = 4 - length;                      // how many blanks do we have?

  if (length > 4) return;                           // if length greater than 4 digits we can't display on a four-digit display!

  for (uint8_t digit = 0; digit < blanks; digit++)  // scroll through each digit to determine what to write to the display
  {
      writeDigit(dsply, digit + 1, 11, 0);          // clear digit wherever there are blanks
  }

  for (uint8_t digit = 0; digit < 4; digit++)       // scroll through each digit to determine what to write to the display
  {
      char ch = string[digit];                      // get the ascii character of the next string segment

      if (ch == '-') {
      writeDigit(dsply, digit + 1 + blanks, 12, 0); // check if negative sign needed
      } 
      else {                                        // character must be a digit
      ch -= '0';                                    // convert it to an integer
      writeDigit(dsply, digit + 1 + blanks, ch, 0); // write it to the display; right justify the integer
      } 
  }
}

void writeFloat(uint8_t dsply, float data, uint8_t dp)
{
  char string[10] = "";  // define character array to hold the digits
  int datanew = 0;
  
  switch (dp)
  {
    case 0:
    datanew = (int )(1.0f*data);
    break;
 
    case 1:
    datanew = (int )(10.0f*data);
    break;

    case 2:
    datanew = (int )(100.0f*data);
    break;
 
    case 3:
    datanew = (int )(1000.0f*data);
    break;
   }
   
  
  itoa(datanew, string);                                    // get ascii character string representation of the integer to be displayed
  uint8_t length = strlen(string);                          // get the length of the string; number of digits in integer
  uint8_t blanks = 4 - length;                              // how many blanks do we have?

  if (length > 4) return;                                   // if length greater than 4 digits we can't display on a four-digit display!

// scroll through each digit to determine what to write to the display
for (uint8_t digit = 0; digit < blanks; digit++)            // first the blanks
  {
          if( (digit + 1) == (4 - dp) ) {                   // handle special case where blank coincides with decimal point
            writeDigit(dsply, digit + 1, 0, 0x80);          // add leading zero before decimal place
          }
          else {
            writeDigit(dsply, digit + 1, 11, 0x00);         // otherwise clear digit wherever there are blanks
          }
  }

  for (uint8_t digit = 0; digit < 4; digit++)               // now the characters to determine what to write to the display
  {
      char ch = string[digit];                              // get the ascii character of the next string segment

      if (ch == '-') {
        if((digit + 1 + blanks) == (4 - dp) ) {
          writeDigit(dsply, digit + 1 + blanks,  0, 0x80);  // check if negative sign needed, add a decimal point
          writeDigit(dsply, digit + 0 + blanks, 12, 0x00);  // add a leading zero
        }
        else {
          writeDigit(dsply, digit + 1 + blanks, 12, 0x00);  // check if negative sign needed, no decimal point
        }
        }
      else  {                                               // character must be a digit
        ch -= '0';                                          // convert it to an integer
        if((digit + 1 + blanks) == (4 - dp) ) {
          writeDigit(dsply, digit + 1 + blanks, ch, 0x80);  // write it to the display with decimal point; right justify the integer
        } 
        else {
          writeDigit(dsply, digit + 1 + blanks, ch, 0x00);  // write it to the display; right justify the integer
        } 
     }
  }
}
  

void writeDigit(uint8_t dsply, uint8_t digit, uint8_t data, uint8_t dp) 
{
if(dsply == 1) {
  digit = (digit - 1)*2 + 0; 
} 
if(dsply == 2) {
  digit = (digit - 1)*2 + 8 ;
}
if(dsply == 3) {
  digit = (digit - 1)*2 + 1;
}
if(dsply == 4) {
  digit = (digit - 1)*2 + 9;
}
writeByte(HT16K33_ADDRESS, digit, numberTable[data] | dp);
}


void clearDsplay(int dsply) 
{
  for(int i = 0; i < 8; i++)  {
  writeDigit(dsply, i, 11, 0);  // Clear display, 11 is blank in the numberTable above
  }
}


void initHT16K33()
{
  writeCommand(HT16K33_ADDRESS, HT16K33_ON);         // Turn on system oscillator
  writeCommand(HT16K33_ADDRESS, HT16K33_DISPLAYON);  // Display on
  writeCommand(HT16K33_ADDRESS, HT16K33_DIM);        // Set brightness

}


void blinkHT16K33(int time) 
{
  writeCommand(HT16K33_ADDRESS, HT16K33_BLINKON);  // Turn on blink
  wait(time);
  writeCommand(HT16K33_ADDRESS, HT16K33_BLINKOFF);  // Turn on blink
}


 /* itoa:  convert n to characters in s */
 void itoa(int n, char s[])
 {
     int i, sign;
 
     if ((sign = n) < 0)  /* record sign */
         n = -n;          /* make n positive */
     i = 0;
     do {       /* generate digits in reverse order */
         s[i++] = n % 10 + '0';   /* get next digit */
     } while ((n /= 10) > 0);     /* delete it */
     if (sign < 0)
         s[i++] = '-';
     s[i] = '\0';
     reverse(s);
 } 
 
 /* reverse:  reverse string s in place */
 void reverse(char s[])
 {
     int i, j;
     char c;
 
     for (i = 0, j = strlen(s)-1; i<j; i++, j--) {
         c = s[i];
         s[i] = s[j];
         s[j] = c;
     }
 }

  // Wire.h read and write protocols
  void writeCommand(uint8_t address, uint8_t command)
  {
   char data_write[1];
   data_write[0] = command;
   i2c.write(address, data_write, 1, 0);
}

   void writeByte(uint8_t address, uint8_t subAddress, uint8_t data)
{
   char data_write[2];
   data_write[0] = subAddress;
   data_write[1] = data;
   i2c.write(address, data_write, 2, 0);
}

    char readByte(uint8_t address, uint8_t subAddress)
{
    char data[1]; // `data` will store the register data     
    char data_write[1];
    data_write[0] = subAddress;
    i2c.write(address, data_write, 1, 1); // no stop
    i2c.read(address, data, 1, 0); 
    return data[0]; 
}

    void readBytes(uint8_t address, uint8_t subAddress, uint8_t count, uint8_t * dest)
{     
    char data[14];
    char data_write[1];
    data_write[0] = subAddress;
    i2c.write(address, data_write, 1, 1); // no stop
    i2c.read(address, data, count, 0); 
    for(int ii = 0; ii < count; ii++) {
     dest[ii] = data[ii];
    }
} 
 };
#endif